
/** JUSH - JavaScript Syntax Highlighter
* @link https://jush.sourceforge.io/
* @author Jakub Vrana, https://www.vrana.cz
* @copyright 2007 Jakub Vrana
* @license https://www.apache.org/licenses/LICENSE-2.0 Apache License, Version 2.0
*/

/* Limitations:
<style> and <script> supposes CDATA or HTML comments
unnecessary escaping (e.g. echo "\'" or ='&quot;') is removed
*/

var jush = {
	create_links: true, // string for extra <a> parameters, e.g. 'target="_blank"'
	timeout: 1000, // milliseconds
	custom_links: { }, // { state: { url: regexp } }, for example { php : { 'doc/$&.html': /\b(getData|setData)\b/g } }
	api: { }, // { state: { function: description } }, for example { php: { array: 'Create an array' } }

	php: /<\?(?!xml)(?:php)?|<script\s+language\s*=\s*(?:"php"|'php'|php)\s*>/i, // asp_tags=0, short_open_tag=1
	num: /(?:0x[0-9a-f]+)|(?:\b[0-9]+\.?[0-9]*|\.[0-9]+)(?:e[+-]?[0-9]+)?/i,

	regexps: undefined,
	subpatterns: { },

	/** Link stylesheet
	* @param string
	* @param [string]
	*/
	style: function (href, media) {
		var link = document.createElement('link');
		link.rel = 'stylesheet';
		if (media) {
			link.media = media;
		}
		link.href = href;
		document.getElementsByTagName('head')[0].appendChild(link);
	},

	/** Highlight text
	* @param string
	* @param string
	* @return string
	*/
	highlight: function (language, text) {
		this.last_tag = '';
		this.last_class = '';
		return this.highlight_states([ language ], text.replace(/\r\n?/g, '\n'), !/^(htm|tag|xml|txt)$/.test(language))[0];
	},

	/** Highlight html
	* @param string
	* @param string
	* @return string
	*/
	highlight_html: function (language, html) {
		var original = html.replace(/<br(\s+[^>]*)?>/gi, '\n');
		var highlighted = jush.highlight(language, jush.html_entity_decode(original.replace(/<[^>]*>/g, '')));

		var inject = { };
		var pos = 0;
		var last_offset = 0;
		original.replace(/(&[^;]+;)|(?:<[^>]+>)+/g, function (str, entity, offset) {
			pos += (offset - last_offset) + (entity ? 1 : 0);
			if (!entity) {
				inject[pos] = str;
			}
			last_offset = offset + str.length;
		});

		pos = 0;
		highlighted = highlighted.replace(/([^&<]*)(?:(&[^;]+;)|(?:<[^>]+>)+|$)/g, function (str, text, entity) {
			for (var i = text.length; i >= 0; i--) {
				if (inject[pos + i]) {
					str = str.substr(0, i) + inject[pos + i] + str.substr(i);
					delete inject[pos + i];
				}
			}
			pos += text.length + (entity ? 1 : 0);
			return str;
		});
		return highlighted;
	},

	/** Highlight text in tags
	* @param mixed tag name or array of HTMLElement
	* @param number number of spaces for tab, 0 for tab itself, defaults to 4
	*/
	highlight_tag: function (tag, tab_width) {
		var pre = (typeof tag == 'string' ? document.getElementsByTagName(tag) : tag);
		var tab = '';
		for (var i = (tab_width !== undefined ? tab_width : 4); i--; ) {
			tab += ' ';
		}
		var i = 0;
		var highlight = function () {
			var start = new Date();
			while (i < pre.length) {
				var match = /(^|\s)(?:jush|language(?=-\S))($|\s|-(\S+))/.exec(pre[i].className); // https://www.w3.org/TR/html5/text-level-semantics.html#the-code-element
				if (match) {
					var language = match[3] ? match[3] : 'htm';
					pre[i].innerHTML = '<span class="jush"><span class="jush-' + language + '">' + jush.highlight_html(language, pre[i].innerHTML.replace(/\t/g, tab.length ? tab : '\t')) + '</span></span>'; // span - enable style for class="language-"
				}
				i++;
				if (jush.timeout && window.setTimeout && (new Date() - start) > jush.timeout) {
					window.setTimeout(highlight, 100);
					break;
				}
			}
		};
		highlight();
	},

	link_manual: function (language, text) {
		var code = document.createElement('code');
		code.innerHTML = this.highlight(language, text);
		var as = code.getElementsByTagName('a');
		for (var i = 0; i < as.length; i++) {
			if (as[i].href) {
				return as[i].href;
			}
		}
		return '';
	},

	create_link: function (link, s, attrs) {
		return '<a'
			+ (this.create_links && link ? ' href="' + link + '" class="jush-help"' : '')
			+ (typeof this.create_links == 'string' ? ' ' + this.create_links.replace(/^\s+/, '') : '')
			+ (attrs || '')
			+ '>' + s + '</a>'
		;
	},

	keywords_links: function (state, s) {
		if (/^js(_write|_code)+$/.test(state)) {
			state = 'js';
		}
		if (/^(php_quo_var|php_php|php_sql|php_sqlite|php_pgsql|php_mssql|php_oracle|php_echo|php_phpini|php_http|php_mail)$/.test(state)) {
			state = 'php2';
		}
		if (state == 'sql_code') {
			state = 'sql';
		}
		if (this.links2 && this.links2[state]) {
			var url = this.urls[state];
			var links2 = this.links2[state];
			s = s.replace(links2, function (str, match1) {
				for (var i=arguments.length - 4; i > 1; i--) {
					if (arguments[i]) {
						var link = (/^https?:/.test(url[i-1]) || !url[i-1] ? url[i-1] : url[0].replace(/\$key/g, url[i-1]));
						switch (state) {
							case 'php': link = link.replace(/\$1/g, arguments[i].toLowerCase()); break;
							case 'php_new': link = link.replace(/\$1/g, arguments[i].toLowerCase()).replace(/\\/g, '-'); break; // toLowerCase() - case sensitive after #
							case 'phpini': link = link.replace(/\$1/g, (/^suhosin\./.test(arguments[i])) ? arguments[i] : arguments[i].toLowerCase().replace(/_/g, '-')); break;
							case 'php_doc': link = link.replace(/\$1/g, arguments[i].replace(/^\W+/, '')); break;
							case 'js_doc': link = link.replace(/\$1/g, arguments[i].replace(/^\W*(.)/, function (match, p1) { return p1.toUpperCase(); })); break;
							case 'http': link = link.replace(/\$1/g, arguments[i].toLowerCase()); break;
							case 'sql': link = link.replace(/\$1/g, arguments[i].replace(/\b(ALTER|CREATE|DROP|RENAME|SHOW)\s+SCHEMA\b/, '$1 DATABASE').toLowerCase().replace(/\s+|_/g, '-')); break;
							case 'sqlset': link = link.replace(/\$1/g, (links2.test(arguments[i].replace(/_/g, '-')) ? arguments[i].replace(/_/g, '-') : arguments[i]).toLowerCase()); break;
							case 'sqlstatus': link = link.replace(/\$1/g, (/mariadb/.test(url[0]) ? arguments[i].toLowerCase() : arguments[i])); break;
							case 'sqlite': link = link.replace(/\$1/g, arguments[i].toLowerCase().replace(/\s+/g, '')); break;
							case 'sqliteset': link = link.replace(/\$1/g, arguments[i].toLowerCase()); break;
							case 'sqlitestatus': link = link.replace(/\$1/g, arguments[i].toLowerCase()); break;
							case 'pgsql': link = link.replace(/\$1/g, arguments[i].toLowerCase().replace(/\s+/g, (i == 1 ? '-' : ''))); break;
							case 'pgsqlset': link = link.replace(/\$1/g, arguments[i].replace(/_/g, '-').toUpperCase()); break;
							case 'cnf': link = link.replace(/\$1/g, arguments[i].toLowerCase()); break;
							case 'js': link = link.replace(/\$1/g, arguments[i].replace(/\./g, '/')); break;
							default: link = link.replace(/\$1/g, arguments[i]).replace(/\\/g, '-');
						}
						var title = '';
						if (jush.api[state]) {
							title = jush.api[state][(state == 'js' ? arguments[i] : arguments[i].toLowerCase())];
						}
						return (match1 ? match1 : '') + jush.create_link(link, arguments[i], (title ? ' title="' + jush.htmlspecialchars_quo(title) + '"' : '')) + (arguments[arguments.length - 3] ? arguments[arguments.length - 3] : '');
					}
				}
			});
		}
		if (this.custom_links[state]) {
			if (Array.isArray(this.custom_links[state])) { // backwards compatibility
				var url = this.custom_links[state][0];
				var re = this.custom_links[state][1];
				this.custom_links[state] = {};
				this.custom_links[state][url] = re;
			}
			for (var url in this.custom_links[state]) {
				s = s.replace(this.custom_links[state][url], function (str) {
					var offset = arguments[arguments.length - 2];
					if (/<[^>]*$/.test(s.substr(0, offset))) {
						return str; // don't create links inside tags
					}
					return '<a href="' + jush.htmlspecialchars_quo(url.replace('$&', encodeURIComponent(str))) + '" class="jush-custom">' + str + '</a>' // not create_link() - ignores create_links
				});
			}
		}
		return s;
	},

	build_regexp: function (key, tr1) {
		var re = [ ];
		subpatterns = [ '' ];
		for (var k in tr1) {
			var in_bra = false;
			subpatterns.push(k);
			var s = tr1[k].source.replace(/\\.|\((?!\?)|\[|]|([a-z])(?:-([a-z]))?/gi, function (str, match1, match2) {
				// count capturing subpatterns
				if (str == (in_bra ? ']' : '[')) {
					in_bra = !in_bra;
				}
				if (str == '(') {
					subpatterns.push(k);
				}
				if (match1 && tr1[k].ignoreCase) {
					if (in_bra) {
						return str.toLowerCase() + str.toUpperCase();
					}
					return '[' + match1.toLowerCase() + match1.toUpperCase() + ']' + (match2 ? '-[' + match2.toLowerCase() + match2.toUpperCase() + ']' : '');
				}
				return str;
			});
			re.push('(' + s + ')');
		}
		this.subpatterns[key] = subpatterns;
		this.regexps[key] = new RegExp(re.join('|'), 'g');
	},

	highlight_states: function (states, text, in_php, escape) {
		if (!this.regexps) {
			this.regexps = { };
			for (var key in this.tr) {
				this.build_regexp(key, this.tr[key]);
			}
		} else {
			for (var key in this.tr) {
				this.regexps[key].lastIndex = 0;
			}
		}
		var state = states[states.length - 1];
		if (!this.tr[state]) {
			return [ this.htmlspecialchars(text), states ];
		}
		var ret = [ ]; // return
		for (var i=1; i < states.length; i++) {
			ret.push('<span class="jush-' + states[i] + '">');
		}
		var match;
		var child_states = [ ];
		var s_states;
		var start = 0;
		while (start < text.length && (match = this.regexps[state].exec(text))) {
			if (states[0] != 'htm' && /^<\/(script|style)>$/i.test(match[0])) {
				continue;
			}
			var key, m = [ ];
			for (var i = match.length; i--; ) {
				if (match[i] || !match[0].length) { // WScript returns empty string even for non matched subexpressions
					key = this.subpatterns[state][i];
					while (this.subpatterns[state][i - 1] == key) {
						i--;
					}
					while (this.subpatterns[state][i] == key) {
						m.push(match[i]);
						i++;
					}
					break;
				}
			}
			if (!key) {
				return [ 'regexp not found', [ ] ];
			}

			if (in_php && key == 'php') {
				continue;
			}
			//~ console.log(states + ' (' + key + '): ' + text.substring(start).replace(/\n/g, '\\n'));
			var out = (key.charAt(0) == '_');
			var division = match.index + (key == 'php_halt2' ? match[0].length : 0);
			var s = text.substring(start, division);

			// highlight children
			var prev_state = states[states.length - 2];
			if (/^(att_quo|att_apo|att_val)$/.test(state) && (/^(att_js|att_css|att_http)$/.test(prev_state) || /^\s*javascript:/i.test(s))) { // javascript: - easy but without own state //! should be checked only in %URI;
				child_states.unshift(prev_state == 'att_css' ? 'css_pro' : (prev_state == 'att_http' ? 'http' : 'js'));
				s_states = this.highlight_states(child_states, this.html_entity_decode(s), true, (state == 'att_apo' ? this.htmlspecialchars_apo : (state == 'att_quo' ? this.htmlspecialchars_quo : this.htmlspecialchars_quo_apo)));
			} else if (state == 'css_js' || state == 'cnf_http' || state == 'cnf_phpini' || state == 'sql_sqlset' || state == 'sqlite_sqliteset' || state == 'pgsql_pgsqlset') {
				child_states.unshift(state.replace(/^[^_]+_/, ''));
				s_states = this.highlight_states(child_states, s, true);
			} else if ((state == 'php_quo' || state == 'php_apo') && /^(php_php|php_sql|php_sqlite|php_pgsql|php_mssql|php_oracle|php_phpini|php_http|php_mail)$/.test(prev_state)) {
				child_states.unshift(prev_state.substr(4));
				s_states = this.highlight_states(child_states, this.stripslashes(s), true, (state == 'php_apo' ? this.addslashes_apo : this.addslashes_quo));
			} else if (key == 'php_halt2') {
				child_states.unshift('htm');
				s_states = this.highlight_states(child_states, s, true);
			} else if ((state == 'apo' || state == 'quo') && prev_state == 'js_write_code') {
				child_states.unshift('htm');
				s_states = this.highlight_states(child_states, s, true);
			} else if ((state == 'apo' || state == 'quo') && prev_state == 'js_http_code') {
				child_states.unshift('http');
				s_states = this.highlight_states(child_states, s, true);
			} else if (((state == 'php_quo' || state == 'php_apo') && prev_state == 'php_echo') || (state == 'php_eot2' && states[states.length - 3] == 'php_echo')) {
				var i;
				for (i=states.length; i--; ) {
					prev_state = states[i];
					if (prev_state.substring(0, 3) != 'php' && prev_state != 'att_quo' && prev_state != 'att_apo' && prev_state != 'att_val') {
						break;
					}
					prev_state = '';
				}
				var f = (state == 'php_eot2' ? this.addslashes : (state == 'php_apo' ? this.addslashes_apo : this.addslashes_quo));
				s = this.stripslashes(s);
				if (/^(att_js|att_css|att_http)$/.test(prev_state)) {
					var g = (states[i+1] == 'att_quo' ? this.htmlspecialchars_quo : (states[i+1] == 'att_apo' ? this.htmlspecialchars_apo : this.htmlspecialchars_quo_apo));
					child_states.unshift(prev_state == 'att_js' ? 'js' : prev_state.substr(4));
					s_states = this.highlight_states(child_states, this.html_entity_decode(s), true, function (string) { return f(g(string)); });
				} else if (prev_state && child_states) {
					child_states.unshift(prev_state);
					s_states = this.highlight_states(child_states, s, true, f);
				} else {
					s = this.htmlspecialchars(s);
					s_states = [ (escape ? escape(s) : s), (!out || !/^(att_js|att_css|att_http|css_js|js_write_code|js_http_code|php_php|php_sql|php_sqlite|php_pgsql|php_mssql|php_oracle|php_echo|php_phpini|php_http|php_mail)$/.test(state) ? child_states : [ ]) ];
				}
			} else {
				s = this.htmlspecialchars(s);
				s_states = [ (escape ? escape(s) : s), (!out || !/^(att_js|att_css|att_http|css_js|js_write_code|js_http_code|php_php|php_sql|php_sqlite|php_pgsql|php_mssql|php_oracle|php_echo|php_phpini|php_http|php_mail)$/.test(state) ? child_states : [ ]) ]; // reset child states when leaving construct
			}
			s = s_states[0];
			child_states = s_states[1];
			s = this.keywords_links(state, s);
			ret.push(s);

			s = text.substring(division, match.index + match[0].length);
			s = (m.length < 3 ? (s ? '<span class="jush-op">' + this.htmlspecialchars(escape ? escape(s) : s) + '</span>' : '') : (m[1] ? '<span class="jush-op">' + this.htmlspecialchars(escape ? escape(m[1]) : m[1]) + '</span>' : '') + this.htmlspecialchars(escape ? escape(m[2]) : m[2]) + (m[3] ? '<span class="jush-op">' + this.htmlspecialchars(escape ? escape(m[3]) : m[3]) + '</span>' : ''));
			if (!out) {
				if (this.links && this.links[key] && m[2]) {
					if (/^tag/.test(key)) {
						this.last_tag = m[2].toLowerCase();
					}
					var link = m[2].toLowerCase();
					var k_link = '';
					for (var k in this.links[key]) {
						var m2 = this.links[key][k].exec(m[2]);
						if (m2) {
							if (m2[1]) {
								link = m2[1].toLowerCase().replace(/\\/g, '-'); // \ is PHP namespace;
							}
							k_link = k;
							if (key != 'att') {
								break;
							}
						}
					}
					if (key == 'php_met') {
						this.last_class = (k_link && !/^(self|parent|static|dir)$/i.test(link) ? link : '');
					}
					if (k_link) {
						s = (m[1] ? '<span class="jush-op">' + this.htmlspecialchars(escape ? escape(m[1]) : m[1]) + '</span>' : '');
						s += this.create_link(
							(/^https?:/.test(k_link) ? k_link : this.urls[key].replace(/\$key/, k_link))
								.replace(/\$val/, (/^https?:/.test(k_link) ? link.toLowerCase() : link))
								.replace(/\$tag/, this.last_tag),
							this.htmlspecialchars(escape ? escape(m[2]) : m[2])); //! use jush.api
						s += (m[3] ? '<span class="jush-op">' + this.htmlspecialchars(escape ? escape(m[3]) : m[3]) + '</span>' : '');
					}
				}
				ret.push('<span class="jush-' + key + '">', s);
				states.push(key);
				if (state == 'php_eot') {
					this.tr.php_eot2._2 = new RegExp('(\n)(' + match[1] + ')(;?\n)');
					this.build_regexp('php_eot2', (match[2] == "'" ? { _2: this.tr.php_eot2._2 } : this.tr.php_eot2));
				} else if (state == 'pgsql_eot') {
					this.tr.pgsql_eot2._2 = new RegExp('\\$' + match[0].replace(/\$/, '\\$'));
					this.build_regexp('pgsql_eot2', this.tr.pgsql_eot2);
				}
			} else {
				if (state == 'php_met' && this.last_class) {
					var title = (jush.api['php2'] ? jush.api['php2'][(this.last_class + '::' + s).toLowerCase()] : '');
					s = this.create_link(this.urls[state].replace(/\$key/, this.last_class) + '.' + s.toLowerCase(), s, (title ? ' title="' + this.htmlspecialchars_quo(title) + '"' : ''));
				}
				ret.push(s);
				for (var i = Math.min(states.length, +key.substr(1)); i--; ) {
					ret.push('</span>');
					states.pop();
				}
			}
			start = match.index + match[0].length;
			if (!states.length) { // out of states
				break;
			}
			state = states[states.length - 1];
			this.regexps[state].lastIndex = start;
		}
		ret.push(this.keywords_links(state, this.htmlspecialchars(text.substring(start))));
		for (var i=1; i < states.length; i++) {
			ret.push('</span>');
		}
		states.shift();
		return [ ret.join(''), states ];
	},

	/** Replace <&> by HTML entities
	* @param string
	* @return string
	*/
	htmlspecialchars: function (string) {
		return string.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
	},

	htmlspecialchars_quo: function (string) {
		return jush.htmlspecialchars(string).replace(/"/g, '&quot;'); // jush - this.htmlspecialchars_quo is passed as reference
	},

	htmlspecialchars_apo: function (string) {
		return jush.htmlspecialchars(string).replace(/'/g, '&#39;');
	},

	htmlspecialchars_quo_apo: function (string) {
		return jush.htmlspecialchars_quo(string).replace(/'/g, '&#39;');
	},

	/** Decode HTML entities
	* @param string
	* @return string
	*/
	html_entity_decode: function (string) {
		return string.replace(/&lt;/g, '<').replace(/&gt;/g, '>').replace(/&quot;/g, '"').replace(/&nbsp;/g, '\u00A0').replace(/&#(?:([0-9]+)|x([0-9a-f]+));/gi, function (str, p1, p2) { //! named entities
			return String.fromCharCode(p1 ? p1 : parseInt(p2, 16));
		}).replace(/&amp;/g, '&');
	},

	/** Add backslash before backslash
	* @param string
	* @return string
	*/
	addslashes: function (string) {
		return string.replace(/\\/g, '\\$&');
	},

	addslashes_apo: function (string) {
		return string.replace(/[\\']/g, '\\$&');
	},

	addslashes_quo: function (string) {
		return string.replace(/[\\"]/g, '\\$&');
	},

	/** Remove backslash before \"'
	* @param string
	* @return string
	*/
	stripslashes: function (string) {
		return string.replace(/\\([\\"'])/g, '$1');
	}
};



jush.tr = { // transitions - key: go inside this state, _2: go outside 2 levels (number alone is put to the beginning in Chrome)
	// regular expressions matching empty string could be used only in the last key
	quo: { php: jush.php, esc: /\\/, _1: /"/ },
	apo: { php: jush.php, esc: /\\/, _1: /'/ },
	com: { php: jush.php, _1: /\*\// },
	com_nest: { com_nest: /\/\*/, _1: /\*\// },
	php: { _1: /\?>/ }, // overwritten by jush-php.js
	esc: { _1: /./ }, //! php_quo allows [0-7]{1,3} and x[0-9A-Fa-f]{1,2}
	one: { _1: /(?=\n)/ },
	num: { _1: /()/ },

	sql_apo: { esc: /\\/, _0: /''/, _1: /'/ },
	sql_quo: { esc: /\\/, _0: /""/, _1: /"/ },
	sql_var: { _1: /(?=[^_.$a-zA-Z0-9])/ },
	sqlite_apo: { _0: /''/, _1: /'/ },
	sqlite_quo: { _0: /""/, _1: /"/ },
	bac: { _1: /`/ },
	bra: { _1: /]/ }
};

// string: $key stands for key in jush.links, $val stands for found string
// array: [0] is base, other elements correspond to () in jush.links2, $key stands for text of selected element, $1 stands for found string
jush.urls = { };
jush.links = { };
jush.links2 = { }; // first and last () is used as delimiter
jush.tr.oracle = { sqlite_apo: /n?'/i, sqlite_quo: /"/, one: /--/, com: /\/\*/, num: /(?:\b[0-9]+\.?[0-9]*|\.[0-9]+)(?:e[+-]?[0-9]+)?[fd]?/i }; //! q'

jush.urls.oracle = ['https://download.oracle.com/docs/cd/B19306_01/server.102/b14200/$key',
	'statements_1003.htm', 'statements_1004.htm', 'statements_1005.htm', 'statements_1006.htm', 'statements_1007.htm', 'statements_1008.htm', 'statements_1009.htm', 'statements_1010.htm', 'statements_2001.htm', 'statements_2002.htm', 'statements_2003.htm', 'statements_2004.htm', 'statements_2005.htm', 'statements_2006.htm', 'statements_2007.htm', 'statements_2008.htm', 'statements_2009.htm', 'statements_2010.htm', 'statements_2011.htm', 'statements_2012.htm', 'statements_2013.htm', 'statements_3001.htm', 'statements_3002.htm', 'statements_4001.htm', 'statements_4002.htm', 'statements_4003.htm', 'statements_4004.htm', 'statements_4005.htm', 'statements_4006.htm', 'statements_4007.htm', 'statements_4008.htm', 'statements_4009.htm', 'statements_4010.htm', 'statements_5001.htm', 'statements_5002.htm', 'statements_5003.htm', 'statements_5004.htm', 'statements_5005.htm', 'statements_5006.htm', 'statements_5007.htm', 'statements_5008.htm', 'statements_5009.htm', 'statements_5010.htm', 'statements_5011.htm', 'statements_5012.htm', 'statements_6001.htm', 'statements_6002.htm', 'statements_6003.htm', 'statements_6004.htm', 'statements_6005.htm', 'statements_6006.htm', 'statements_6007.htm', 'statements_6008.htm', 'statements_6009.htm', 'statements_6010.htm', 'statements_6011.htm', 'statements_6012.htm', 'statements_6013.htm', 'statements_6014.htm', 'statements_6015.htm', 'statements_6016.htm', 'statements_7001.htm', 'statements_7002.htm', 'statements_7003.htm', 'statements_7004.htm', 'statements_8001.htm', 'statements_8002.htm', 'statements_8003.htm', 'statements_8004.htm', 'statements_8005.htm', 'statements_8006.htm', 'statements_8007.htm', 'statements_8008.htm', 'statements_8009.htm', 'statements_8010.htm', 'statements_8011.htm', 'statements_8012.htm', 'statements_8013.htm', 'statements_8014.htm', 'statements_8015.htm', 'statements_8016.htm', 'statements_8017.htm', 'statements_8018.htm', 'statements_8019.htm', 'statements_8020.htm', 'statements_8021.htm', 'statements_8022.htm', 'statements_8023.htm', 'statements_8024.htm', 'statements_8025.htm', 'statements_8026.htm', 'statements_8027.htm', 'statements_8028.htm', 'statements_9001.htm', 'statements_9002.htm', 'statements_9003.htm', 'statements_9004.htm', 'statements_9005.htm', 'statements_9006.htm', 'statements_9007.htm', 'statements_9008.htm', 'statements_9009.htm', 'statements_9010.htm', 'statements_9011.htm', 'statements_9012.htm', 'statements_9013.htm', 'statements_9014.htm', 'statements_9015.htm', 'statements_9016.htm', 'statements_9017.htm', 'statements_9018.htm', 'statements_9019.htm', 'statements_9020.htm', 'statements_9021.htm', 'statements_10001.htm', 'statements_10002.htm', 'statements_10003.htm', 'statements_10004.htm', 'statements_10005.htm', 'statements_10006.htm', 'statements_10007.htm',
	'functions002.htm', 'functions003.htm', 'functions004.htm', 'functions005.htm', 'functions006.htm', 'functions007.htm', 'functions008.htm', 'functions009.htm', 'functions010.htm', 'functions011.htm', 'functions012.htm', 'functions013.htm', 'functions014.htm', 'functions015.htm', 'functions016.htm', 'functions017.htm', 'functions018.htm', 'functions019.htm', 'functions020.htm', 'functions021.htm', 'functions022.htm', 'functions023.htm', 'functions024.htm', 'functions025.htm', 'functions026.htm', 'functions027.htm', 'functions028.htm', 'functions029.htm#i1279881', 'functions029.htm#i1281694', 'functions030.htm', 'functions031.htm', 'functions032.htm', 'functions033.htm', 'functions034.htm', 'functions035.htm', 'functions036.htm', 'functions037.htm', 'functions038.htm', 'functions039.htm', 'functions040.htm', 'functions041.htm', 'functions042.htm', 'functions043.htm', 'functions044.htm', 'functions045.htm', 'functions046.htm', 'functions047.htm', 'functions048.htm', 'functions049.htm', 'functions050.htm', 'functions052.htm', 'functions053.htm', 'functions054.htm', 'functions055.htm', 'functions056.htm', 'functions057.htm', 'functions058.htm', 'functions059.htm', 'functions060.htm', 'functions061.htm', 'functions062.htm', 'functions063.htm', 'functions064.htm', 'functions065.htm', 'functions066.htm', 'functions067.htm', 'functions068.htm', 'functions069.htm', 'functions070.htm', 'functions071.htm', 'functions072.htm', 'functions073.htm', 'functions074.htm', 'functions075.htm', 'functions076.htm', 'functions077.htm', 'functions078.htm', 'functions079.htm', 'functions080.htm', 'functions081.htm', 'functions082.htm', 'functions083.htm', 'functions084.htm', 'functions085.htm', 'functions086.htm', 'functions087.htm', 'functions088.htm', 'functions089.htm', 'functions090.htm', 'functions091.htm', 'functions092.htm', 'functions093.htm', 'functions094.htm', 'functions095.htm', 'functions096.htm', 'functions097.htm', 'functions098.htm', 'functions099.htm', 'functions100.htm', 'functions101.htm', 'functions102.htm', 'functions103.htm', 'functions104.htm', 'functions105.htm', 'functions106.htm', 'functions107.htm', 'functions108.htm', 'functions109.htm', 'functions110.htm', 'functions111.htm', 'functions112.htm', 'functions113.htm', 'functions114.htm', 'functions115.htm', 'functions116.htm', 'functions117.htm', 'functions118.htm', 'functions119.htm', 'functions120.htm', 'functions121.htm', 'functions122.htm', 'functions123.htm', 'functions124.htm', 'functions125.htm', 'functions126.htm', 'functions127.htm', 'functions128.htm', 'functions129.htm', 'functions130.htm', 'functions131.htm', 'functions132.htm', 'functions133.htm', 'functions134.htm', 'functions135.htm', 'functions137.htm', 'functions138.htm', 'functions139.htm', 'functions140.htm', 'functions141.htm', 'functions142.htm', 'functions143.htm', 'functions144.htm', 'functions145.htm', 'functions146.htm', 'functions147.htm', 'functions148.htm', 'functions149.htm', 'functions150.htm', 'functions151.htm', 'functions152.htm', 'functions153.htm', 'functions154.htm', 'functions155.htm', 'functions156.htm', 'functions157.htm#sthref2125', 'functions157.htm#sthref2129', 'functions157.htm#sthref2132', 'functions158.htm', 'functions159.htm', 'functions160.htm', 'functions161.htm', 'functions162.htm', 'functions163.htm', 'functions164.htm', 'functions165.htm', 'functions166.htm', 'functions167.htm', 'functions168.htm', 'functions169.htm', 'functions170.htm', 'functions171.htm', 'functions172.htm', 'functions173.htm', 'functions174.htm', 'functions175.htm', 'functions176.htm', 'functions177.htm', 'functions178.htm', 'functions179.htm', 'functions182.htm', 'functions183.htm', 'functions184.htm', 'functions185.htm', 'functions186.htm', 'functions187.htm', 'functions190.htm', 'functions191.htm', 'functions192.htm', 'functions193.htm', 'functions194.htm', 'functions195.htm', 'functions196.htm', 'functions198.htm', 'functions199.htm', 'functions200.htm', 'functions202.htm', 'functions203.htm', 'functions204.htm', 'functions205.htm', 'functions206.htm', 'functions207.htm', 'functions208.htm', 'functions209.htm', 'functions210.htm', 'functions211.htm', 'functions212.htm', 'functions213.htm', 'functions214.htm', 'functions215.htm', 'functions216.htm', 'functions217.htm', 'functions218.htm', 'functions219.htm', 'functions220.htm', 'functions221.htm', 'functions222.htm', 'functions223.htm', 'functions224.htm', 'functions225.htm', 'functions226.htm', 'functions227.htm', 'functions228.htm', 'functions229.htm'
];

jush.links2.oracle = /(\b)(ALTER\s+CLUSTER|(ALTER\s+DATABASE)|(ALTER\s+DIMENSION)|(ALTER\s+DISKGROUP)|(ALTER\s+FUNCTION)|(ALTER\s+INDEX)|(ALTER\s+INDEXTYPE)|(ALTER\s+JAVA)|(ALTER\s+MATERIALIZED\s+VIEW)|(ALTER\s+MATERIALIZED\s+VIEW\s+LOG)|(ALTER\s+OPERATOR)|(ALTER\s+OUTLINE)|(ALTER\s+PACKAGE)|(ALTER\s+PROCEDURE)|(ALTER\s+PROFILE)|(ALTER\s+RESOURCE\s+COST)|(ALTER\s+ROLE)|(ALTER\s+ROLLBACK\s+SEGMENT)|(ALTER\s+SEQUENCE)|(ALTER\s+SESSION)|(ALTER\s+SYSTEM)|(ALTER\s+TABLE)|(ALTER\s+TABLESPACE)|(ALTER\s+TRIGGER)|(ALTER\s+TYPE)|(ALTER\s+USER)|(ALTER\s+VIEW)|(ANALYZE)|(ASSOCIATE\s+STATISTICS)|(AUDIT)|(CALL)|(COMMENT)|(COMMIT)|(CREATE\s+CLUSTER)|(CREATE\s+CONTEXT)|(CREATE\s+CONTROLFILE)|(CREATE\s+DATABASE)|(CREATE\s+DATABASE\s+LINK)|(CREATE\s+DIMENSION)|(CREATE\s+DIRECTORY)|(CREATE\s+DISKGROUP)|(CREATE\s+FUNCTION)|(CREATE\s+INDEX)|(CREATE\s+INDEXTYPE)|(CREATE\s+JAVA)|(CREATE\s+LIBRARY)|(CREATE\s+MATERIALIZED\s+VIEW)|(CREATE\s+MATERIALIZED\s+VIEW\s+LOG)|(CREATE\s+OPERATOR)|(CREATE\s+OUTLINE)|(CREATE\s+PACKAGE)|(CREATE\s+PACKAGE\s+BODY)|(CREATE\s+PFILE)|(CREATE\s+PROCEDURE)|(CREATE\s+PROFILE)|(CREATE\s+RESTORE\s+POINT)|(CREATE\s+ROLE)|(CREATE\s+ROLLBACK\s+SEGMENT)|(CREATE\s+SCHEMA)|(CREATE\s+SEQUENCE)|(CREATE\s+SPFILE)|(CREATE\s+SYNONYM)|(CREATE\s+TABLE)|(CREATE\s+TABLESPACE)|(CREATE\s+TRIGGER)|(CREATE\s+TYPE)|(CREATE\s+TYPE\s+BODY)|(CREATE\s+USER)|(CREATE\s+VIEW)|(DELETE)|(DISASSOCIATE\s+STATISTICS)|(DROP\s+CLUSTER)|(DROP\s+CONTEXT)|(DROP\s+DATABASE)|(DROP\s+DATABASE\s+LINK)|(DROP\s+DIMENSION)|(DROP\s+DIRECTORY)|(DROP\s+DISKGROUP)|(DROP\s+FUNCTION)|(DROP\s+INDEX)|(DROP\s+INDEXTYPE)|(DROP\s+JAVA)|(DROP\s+LIBRARY)|(DROP\s+MATERIALIZED\s+VIEW)|(DROP\s+MATERIALIZED\s+VIEW\s+LOG)|(DROP\s+OPERATOR)|(DROP\s+OUTLINE)|(DROP\s+PACKAGE)|(DROP\s+PROCEDURE)|(DROP\s+PROFILE)|(DROP\s+RESTORE\s+POINT)|(DROP\s+ROLE)|(DROP\s+ROLLBACK\s+SEGMENT)|(DROP\s+SEQUENCE)|(DROP\s+SYNONYM)|(DROP\s+TABLE)|(DROP\s+TABLESPACE)|(DROP\s+TRIGGER)|(DROP\s+TYPE)|(DROP\s+TYPE\s+BODY)|(DROP\s+USER)|(DROP\s+VIEW)|(EXPLAIN\s+PLAN)|(FLASHBACK\s+DATABASE)|(FLASHBACK\s+TABLE)|(GRANT)|(INSERT)|(LOCK\s+TABLE)|(MERGE)|(NOAUDIT)|(PURGE)|(RENAME)|(REVOKE)|(ROLLBACK)|(SAVEPOINT)|(SELECT)|(SET\s+CONSTRAINTS?)|(SET\s+ROLE)|(SET\s+TRANSACTION)|(TRUNCATE)|(UPDATE)|(abs)|(acos)|(add_months)|(appendchildxml)|(asciistr)|(ascii)|(asin)|(atan)|(atan2)|(avg)|(bfilename)|(bin_to_num)|(bitand)|(cardinality)|(cast)|(ceil)|(chartorowid)|(chr)|(cluster_id)|(cluster_probability)|(cluster_set)|(coalesce)|(collect)|(compose)|(concat)|(convert)|(corr)|(corr_s)|(corr_k)|(cos)|(cosh)|(count)|(covar_pop)|(covar_samp)|(cume_dist)|(current_date)|(current_timestamp)|(cv)|(dbtimezone)|(decode)|(decompose)|(deletexml)|(dense_rank)|(depth)|(deref)|(dump)|(empty_[bc]lob)|(existsnode)|(exp)|(extract)|(extractvalue)|(feature_id)|(feature_set)|(feature_value)|(first)|(first_value)|(floor)|(from_tz)|(greatest)|(group_id)|(grouping)|(grouping_id)|(hextoraw)|(initcap)|(insertchildxml)|(insertxmlbefore)|(instr)|(iteration_number)|(lag)|(last)|(last_day)|(last_value)|(lead)|(least)|(length)|(ln)|(lnnvl)|(localtimestamp)|(log)|(lower)|(lpad)|(ltrim)|(make_ref)|(max)|(median)|(min)|(mod)|(months_between)|(nanvl)|(nchr)|(new_time)|(next_day)|(nls_charset_decl_len)|(nls_charset_id)|(nls_charset_name)|(nls_initcap)|(nls_lower)|(nlssort)|(nls_upper)|(ntile)|(nullif)|(numtodsinterval)|(numtoyminterval)|(nvl)|(nvl2)|(ora_hash)|(path)|(percent_rank)|(percentile_cont)|(percentile_disc)|(power)|(powermultiset)|(powermultiset_by_cardinality)|(prediction)|(prediction_cost)|(prediction_details)|(prediction_probability)|(prediction_set)|(presentnnv)|(presentv)|(previous)|(rank)|(ratio_to_report)|(rawtohex)|(rawtonhex)|(ref)|(reftohex)|(regexp_instr)|(regexp_replace)|(regexp_substr)|(regr_(?:slope|intercept|count|r2|avgx|avgy|sxx|syy|sxy))|(remainder)|(replace)|(round)|(row_number)|(rowidtochar)|(rowidtonchar)|(rpad)|(rtrim)|(scn_to_timestamp)|(sessiontimezone)|(set)|(sign)|(sin)|(sinh)|(soundex)|(sqrt)|(stats_binomial_test)|(stats_crosstab)|(stats_f_test)|(stats_ks_test)|(stats_mode)|(stats_mw_test)|(stats_one_way_anova)|(stats_t_test_one)|(stats_t_test_paired)|(stats_t_test_indepu?)|(stats_wsr_test)|(stddev)|(stddev_pop)|(stddev_samp)|(substr)|(sum)|(sys_connect_by_path)|(sys_context)|(sys_dburigen)|(sys_extract_utc)|(sys_guid)|(sys_typeid)|(sys_xmlagg)|(sys_xmlgen)|(sysdate)|(systimestamp)|(tan)|(tanh)|(timestamp_to_scn)|(to_binary_double)|(to_binary_float)|(to_char)|(to_clob)|(to_date)|(to_dsinterval)|(to_lob)|(to_multi_byte)|(to_nchar)|(to_nclob)|(to_number)|(to_single_byte)|(to_timestamp)|(to_timestamp_tz)|(to_yminterval)|(translate)|(treat)|(trim)|(trunc)|(tz_offset)|(uid)|(unistr)|(updatexml)|(upper)|(user)|(userenv)|(value)|(var_pop)|(var_samp)|(variance)|(vsize)|(width_bucket)|(xmlagg)|(xmlcdata)|(xmlcolattval)|(xmlcomment)|(xmlconcat)|(xmlelement)|(xmlforest)|(xmlparse)|(xmlpi)|(xmlquery)|(xmlroot)|(xmlsequence)|(xmlserialize)|(xmltable)|(xmltransform))(\b)/gi; // collisions: IDENTITY, extract, round, to_char, to_nchar, translate, trunc
jush.textarea = (function () {
	//! IE sometimes inserts empty <p> in start of a string when newline is entered inside
	
	function findSelPos(pre) {
		var sel = getSelection();
		if (sel.rangeCount) {
			var range = sel.getRangeAt(0);
			return findPosition(pre, range.startContainer, range.startOffset);
		}
	}

	function findPosition(el, container, offset) {
		var pos = { pos: 0 };
		findPositionRecurse(el, container, offset, pos);
		return pos.pos;
	}

	function findPositionRecurse(child, container, offset, pos) {
		if (child.nodeType == 3) {
			if (child == container) {
				pos.pos += offset;
				return true;
			}
			pos.pos += child.textContent.length;
		} else if (child == container) {
			for (var i = 0; i < offset; i++) {
				findPositionRecurse(child.childNodes[i], container, offset, pos);
			}
			return true;
		} else {
			if (/^(br|div)$/i.test(child.tagName)) {
				pos.pos++;
			}
			for (var i = 0; i < child.childNodes.length; i++) {
				if (findPositionRecurse(child.childNodes[i], container, offset, pos)) {
					return true;
				}
			}
			if (/^p$/i.test(child.tagName)) {
				pos.pos++;
			}
		}
	}
	
	function findOffset(el, pos) {
		return findOffsetRecurse(el, { pos: pos });
	}
	
	function findOffsetRecurse(child, pos) {
		if (child.nodeType == 3) { // 3 - TEXT_NODE
			if (child.textContent.length >= pos.pos) {
				return { container: child, offset: pos.pos };
			}
			pos.pos -= child.textContent.length;
		} else {
			for (var i = 0; i < child.childNodes.length; i++) {
				if (/^br$/i.test(child.childNodes[i].tagName)) {
					if (!pos.pos) {
						return { container: child, offset: i };
					}
					pos.pos--;
					if (!pos.pos && i == child.childNodes.length - 1) { // last invisible <br>
						return { container: child, offset: i };
					}
				} else {
					var result = findOffsetRecurse(child.childNodes[i], pos);
					if (result) {
						return result;
					}
				}
			}
		}
	}
	
	function setSelPos(pre, pos) {
		if (pos) {
			var start = findOffset(pre, pos);
			if (start) {
				var range = document.createRange();
				range.setStart(start.container, start.offset);
				var sel = getSelection();
				sel.removeAllRanges();
				sel.addRange(range);
			}
		}
	}

	function setText(pre, text, end) {
		var lang = 'txt';
		if (text.length < 1e4) { // highlighting is slow with most languages
			var match = /(^|\s)(?:jush|language)-(\S+)/.exec(pre.jushTextarea.className);
			lang = (match ? match[2] : 'htm');
		}
		var html = jush.highlight(lang, text).replace(/\n/g, '<br>');
		setHTML(pre, html, text, end);
		if (openAc) {
			openAutocomplete(pre);
			openAc = false;
		} else {
			closeAutocomplete();
		}
	}
	
	function setHTML(pre, html, text, pos) {
		pre.innerHTML = html;
		pre.lastHTML = pre.innerHTML; // not html because IE reformats the string
		pre.jushTextarea.value = text;
		setSelPos(pre, pos);
	}
	
	function keydown(event) {
		const ctrl = (event.ctrlKey || event.metaKey);
		if (!event.altKey) {
			if (!ctrl && acEl.options.length) {
				const select =
					(event.key == 'ArrowDown' ? Math.min(acEl.options.length - 1, acEl.selectedIndex + 1) :
					(event.key == 'ArrowUp' ? Math.max(0, acEl.selectedIndex - 1) :
					(event.key == 'PageDown' ? Math.min(acEl.options.length - 1, acEl.selectedIndex + acEl.size) :
					(event.key == 'PageUp' ? Math.max(0, acEl.selectedIndex - acEl.size) :
					null))))
				;
				if (select !== null) {
					acEl.selectedIndex = select;
					return false;
				}
				if (/^(Enter|Tab)$/.test(event.key) && !event.shiftKey) {
					insertAutocomplete(this);
					return false;
				}
			}
			
			if (ctrl) {
				if (event.key == ' ') {
					openAutocomplete(this);
				}
			} else if (autocomplete.openBy && (autocomplete.openBy.test(event.key) || event.key == 'Backspace' || (event.key == 'Enter' && event.shiftKey))) {
				openAc = true;
			} else if (/^(Escape|ArrowLeft|ArrowRight|Home|End)$/.test(event.key)) {
				closeAutocomplete();
			}
		}
		
		if (ctrl && !event.altKey) {
			var isUndo = (event.keyCode == 90); // 90 - z
			var isRedo = (event.keyCode == 89 || (event.keyCode == 90 && event.shiftKey)); // 89 - y
			if (isUndo || isRedo) {
				if (isRedo) {
					if (this.jushUndoPos + 1 < this.jushUndo.length) {
						this.jushUndoPos++;
						var undo = this.jushUndo[this.jushUndoPos];
						setText(this, undo.text, undo.end)
					}
				} else if (this.jushUndoPos >= 0) {
					this.jushUndoPos--;
					var undo = this.jushUndo[this.jushUndoPos] || { html: '', text: '' };
					setText(this, undo.text, this.jushUndo[this.jushUndoPos + 1].start);
				}
				return false;
			}
		} else {
			setLastPos(this);
		}
	}
	
	const maxSize = 8;
	const acEl = document.createElement('select');
	acEl.size = maxSize;
	acEl.className = 'jush-autocomplete';
	acEl.style.position = 'absolute';
	acEl.style.zIndex = 1;
	acEl.onclick = () => {
		insertAutocomplete(pre);
	};
	openAc = false;
	closeAutocomplete();

	function findState(node) {
		let match;
		while (node && (!/^(CODE|PRE)$/.test(node.tagName) || !(match = node.className.match(/(^|\s)jush-(\w+)/)))) {
			node = node.parentElement;
		}
		return (match ? match[2] : '');
	}

	function openAutocomplete(pre) {
		const prevSelected = acEl.options[acEl.selectedIndex];
		closeAutocomplete();
		const sel = getSelection();
		if (sel.rangeCount) {
			const range = sel.getRangeAt(0);
			const pos = findSelPos(pre);
			const state = findState(range.startContainer);
			if (state) {
				const ac = autocomplete(
					state,
					pre.innerText.substring(0, pos),
					pre.innerText.substring(pos)
				);
				if (Object.keys(ac).length) {
					let select = 0;
					for (const word in ac) {
						const option = document.createElement('option');
						option.value = ac[word];
						option.textContent = word;
						acEl.append(option);
						if (prevSelected && prevSelected.textContent == word) {
							select = acEl.options.length - 1;
						}
					}
					acEl.selectedIndex = select;
					acEl.size = Math.min(Math.max(acEl.options.length, 2), maxSize);
					positionAutocomplete();
					acEl.style.display = '';
				}
			}
		}
	}
	
	function positionAutocomplete() {
		const sel = getSelection();
		if (sel.rangeCount && acEl.options.length) {
			const pos = findSelPos(pre);
			const range = sel.getRangeAt(0);
			const range2 = range.cloneRange();
			range2.setStart(range.startContainer, Math.max(0, range.startOffset - acEl.options[0].value)); // autocompletions currently couldn't cross container boundary
			const span = document.createElement('span'); // collapsed ranges have empty bounding rect
			range2.insertNode(span);
			acEl.style.left = span.offsetLeft + 'px';
			acEl.style.top = (span.offsetTop + 20) + 'px';
			span.remove();
			setSelPos(pre, pos); // required on iOS
		}
	}
	
	function closeAutocomplete() {
		acEl.options.length = 0;
		acEl.style.display = 'none';
	}
	
	function insertAutocomplete(pre) {
		const sel = getSelection();
		const range = sel.rangeCount && sel.getRangeAt(0);
		if (range) {
			const insert = acEl.options[acEl.selectedIndex].textContent;
			const offset = +acEl.options[acEl.selectedIndex].value;
			forceNewUndo = true;
			pre.lastPos = findSelPos(pre);
			const start = findOffset(pre, pre.lastPos - offset);
			range.setStart(start.container, start.offset);
			document.execCommand('insertText', false, insert);
			openAutocomplete(pre);
		}
	}
	
	function setLastPos(pre) {
		if (pre.lastPos === undefined) {
			pre.lastPos = findSelPos(pre);
		}
	}
	
	var forceNewUndo = true;
	
	function highlight(pre) {
		var start = pre.lastPos;
		pre.lastPos = undefined;
		var innerHTML = pre.innerHTML;
		if (innerHTML != pre.lastHTML) {
			var end = findSelPos(pre);
			innerHTML = innerHTML.replace(/<br>((<\/[^>]+>)*<\/?div>)(?!$)/gi, function (all, rest) {
				if (end) {
					end--;
				}
				return rest;
			});
			pre.innerHTML = innerHTML
				.replace(/<(br|div)\b[^>]*>/gi, '\n') // Firefox, Chrome
				.replace(/&nbsp;(<\/[pP]\b)/g, '$1') // IE
				.replace(/<\/p\b[^>]*>($|<p\b[^>]*>)/gi, '\n') // IE
				.replace(/(&nbsp;)+$/gm, '') // Chrome for some users
			;
			setText(pre, pre.textContent.replace(/\u00A0/g, ' '), end);
			pre.jushUndo.length = pre.jushUndoPos + 1;
			if (forceNewUndo || !pre.jushUndo.length || pre.jushUndo[pre.jushUndoPos].end !== start) {
				pre.jushUndo.push({ text: pre.jushTextarea.value, start: start, end: (forceNewUndo ? undefined : end) });
				pre.jushUndoPos++;
				forceNewUndo = false;
			} else {
				pre.jushUndo[pre.jushUndoPos].text = pre.jushTextarea.value;
				pre.jushUndo[pre.jushUndoPos].end = end;
			}
		}
	}
	
	function input() {
		highlight(this);
	}
	
	function paste(event) {
		if (event.clipboardData) {
			setLastPos(this);
			if (document.execCommand('insertHTML', false, jush.htmlspecialchars(event.clipboardData.getData('text')))) { // Opera doesn't support insertText
				event.preventDefault();
			}
			forceNewUndo = true; // highlighted in input
		}
	}
	
	function click(event) {
		if ((event.ctrlKey || event.metaKey) && event.target.href) {
			open(event.target.href);
		}
		closeAutocomplete();
	}
	
	let pre;
	let autocomplete = () => ({});
	addEventListener('resize', positionAutocomplete);
	
	return function textarea(el, autocompleter) {
		if (!window.getSelection) {
			return;
		}
		if (autocompleter) {
			autocomplete = autocompleter;
		}
		pre = document.createElement('pre');
		pre.contentEditable = true;
		pre.className = el.className + ' jush';
		pre.style.border = '1px inset #ccc';
		pre.style.width = el.clientWidth + 'px';
		pre.style.height = el.clientHeight + 'px';
		pre.style.padding = '3px';
		pre.style.overflow = 'auto';
		pre.style.resize = 'both';
		if (el.wrap != 'off') {
			pre.style.whiteSpace = 'pre-wrap';
		}
		pre.jushTextarea = el;
		pre.jushUndo = [ ];
		pre.jushUndoPos = -1;
		pre.onkeydown = keydown;
		pre.oninput = input;
		pre.onpaste = paste;
		pre.onclick = click;
		pre.appendChild(document.createTextNode(el.value));
		highlight(pre);
		if (el.spellcheck === false) {
			pre.spellcheck = false;
		}
		el.before(pre);
		el.before(acEl);
		if (document.activeElement === el) {
			pre.focus();
			if (!el.value) {
				openAutocomplete(pre);
			}
		}
		acEl.style.font = getComputedStyle(pre).font;
		el.style.display = 'none';
		return pre;
	};
})();
jush.tr.js = { php: jush.php, js_reg: /\s*\/(?![\/*])/, js_obj: /\s*\{/, _1: /}/, js_code: /()/ };
jush.tr.js_code = { php: jush.php, quo: /"/, apo: /'/, js_bac: /`/, js_one: /\/\//, js_doc: /\/\*\*/, com: /\/\*/, num: jush.num, js_write: /(\b)(write(?:ln)?)(\()/, js_http: /(\.)(setRequestHeader|getResponseHeader)(\()/, js: /\{/, _3: /(<)(\/script)(>)/i, _2: /}/, _1: /[^.\])}$\w\s]/ };
jush.tr.js_write = { php: jush.php, js_reg: /\s*\/(?![\/*])/, js_write_code: /()/ };
jush.tr.js_http = { php: jush.php, js_reg: /\s*\/(?![\/*])/, js_http_code: /()/ };
jush.tr.js_write_code = { php: jush.php, quo: /"/, apo: /'/, js_bac: /`/, js_one: /\/\//, com: /\/\*/, num: jush.num, js_write: /\(/, _2: /\)/, _1: /[^\])}$\w\s]/ };
jush.tr.js_http_code = { php: jush.php, quo: /"/, apo: /'/, js_bac: /`/, js_one: /\/\//, com: /\/\*/, num: jush.num, js_http: /\(/, _2: /\)/, _1: /[^\])}$\w\s]/ };
jush.tr.js_one = { php: jush.php, _1: /\n/, _3: /(<)(\/script)(>)/i };
jush.tr.js_reg = { php: jush.php, esc: /\\/, js_reg_bra: /\[/, _1: /\/[a-z]*/i }; //! highlight regexp
jush.tr.js_reg_bra = { php: jush.php, esc: /\\/, _1: /]/ };
jush.tr.js_doc = { _1: /\*\// };
jush.tr.js_arr = { php: jush.php, quo: /"/, apo: /'/, js_bac: /`/, js_one: /\/\//, com: /\/\*/, num: jush.num, js_arr: /\[/, js_obj: /\{/, _1: /]/ };
jush.tr.js_obj = { php: jush.php, js_one: /\s*\/\//, com: /\s*\/\*/, js_val: /:/, _1: /\s*}/, js_key: /()/ };
jush.tr.js_val = { php: jush.php, quo: /"/, apo: /'/, js_bac: /`/, js_one: /\/\//, com: /\/\*/, num: jush.num, js_arr: /\[/, js_obj: /\{/, _1: /,|(?=})/ };
jush.tr.js_key = { php: jush.php, quo: /"/, apo: /'/, js_bac: /`/, js_one: /\/\//, com: /\/\*/, num: jush.num, _1: /(?=[:}])/ };
jush.tr.js_bac = { php: jush.php, esc: /\\/, js: /\$\{/, _1: /`/ };

jush.urls.js_write = 'https://developer.mozilla.org/en/docs/DOM/$key.$val';
jush.urls.js_http = 'https://www.w3.org/TR/XMLHttpRequest/#the-$val-$key';
jush.urls.js = ['https://developer.mozilla.org/en/$key',
	'JavaScript/Reference/Global_Objects/$1',
	'JavaScript/Reference/Statements/$1',
	'JavaScript/Reference/Statements/do...while',
	'JavaScript/Reference/Statements/if...else',
	'JavaScript/Reference/Statements/try...catch',
	'JavaScript/Reference/Operators/Special/$1',
	'DOM/document.$1', 'DOM/element.$1', 'DOM/event.$1', 'DOM/form.$1', 'DOM/table.$1', 'DOM/window.$1',
	'https://www.w3.org/TR/XMLHttpRequest/',
	'JavaScript/Reference/Global_Objects/Array.$1',
	'JavaScript/Reference/Global_Objects/Array$1',
	'JavaScript/Reference/Global_Objects/Date$1',
	'JavaScript/Reference/Global_Objects/Function$1',
	'JavaScript/Reference/Global_Objects/Number$1',
	'JavaScript/Reference/Global_Objects/RegExp$1',
	'JavaScript/Reference/Global_Objects/String$1'
];
jush.urls.js_doc = ['https://code.google.com/p/jsdoc-toolkit/wiki/Tag$key',
	'$1', 'Param', 'Augments', '$1'
];

jush.links.js_write = { 'document': /^(write|writeln)$/ };
jush.links.js_http = { 'method': /^(setRequestHeader|getResponseHeader)$/ };

jush.links2.js = /(\b)(String\.fromCharCode|Date\.(?:parse|UTC)|Math\.(?:E|LN2|LN10|LOG2E|LOG10E|PI|SQRT1_2|SQRT2|abs|acos|asin|atan|atan2|ceil|cos|exp|floor|log|max|min|pow|random|round|sin|sqrt|tan)|Array|Boolean|Date|Error|Function|JavaArray|JavaClass|JavaObject|JavaPackage|Math|Number|Object|Packages|RegExp|String|Infinity|JSON|NaN|undefined|Error|EvalError|RangeError|ReferenceError|SyntaxError|TypeError|URIError|decodeURI|decodeURIComponent|encodeURI|encodeURIComponent|eval|isFinite|isNaN|parseFloat|parseInt|(break|continue|for|function|return|switch|throw|var|while|with)|(do)|(if|else)|(try|catch|finally)|(delete|in|instanceof|new|this|typeof|void)|(alinkColor|anchors|applets|bgColor|body|characterSet|compatMode|contentType|cookie|defaultView|designMode|doctype|documentElement|domain|embeds|fgColor|forms|height|images|implementation|lastModified|linkColor|links|plugins|popupNode|referrer|styleSheets|title|tooltipNode|URL|vlinkColor|width|clear|createAttribute|createDocumentFragment|createElement|createElementNS|createEvent|createNSResolver|createRange|createTextNode|createTreeWalker|evaluate|execCommand|getElementById|getElementsByName|importNode|loadOverlay|queryCommandEnabled|queryCommandIndeterm|queryCommandState|queryCommandValue|write|writeln)|(attributes|childNodes|className|clientHeight|clientLeft|clientTop|clientWidth|dir|firstChild|id|innerHTML|lang|lastChild|localName|name|namespaceURI|nextSibling|nodeName|nodeType|nodeValue|offsetHeight|offsetLeft|offsetParent|offsetTop|offsetWidth|ownerDocument|parentNode|prefix|previousSibling|scrollHeight|scrollLeft|scrollTop|scrollWidth|style|tabIndex|tagName|textContent|addEventListener|appendChild|blur|click|cloneNode|dispatchEvent|focus|getAttribute|getAttributeNS|getAttributeNode|getAttributeNodeNS|getElementsByTagName|getElementsByTagNameNS|hasAttribute|hasAttributeNS|hasAttributes|hasChildNodes|insertBefore|item|normalize|removeAttribute|removeAttributeNS|removeAttributeNode|removeChild|removeEventListener|replaceChild|scrollIntoView|setAttribute|setAttributeNS|setAttributeNode|setAttributeNodeNS|supports|onblur|onchange|onclick|ondblclick|onfocus|onkeydown|onkeypress|onkeyup|onmousedown|onmousemove|onmouseout|onmouseover|onmouseup|onresize)|(altKey|bubbles|button|cancelBubble|cancelable|clientX|clientY|ctrlKey|currentTarget|detail|eventPhase|explicitOriginalTarget|isChar|layerX|layerY|metaKey|originalTarget|pageX|pageY|relatedTarget|screenX|screenY|shiftKey|target|timeStamp|type|view|which|initEvent|initKeyEvent|initMouseEvent|initUIEvent|stopPropagation|preventDefault)|(elements|name|acceptCharset|action|enctype|encoding|method|submit|reset)|(caption|tHead|tFoot|rows|tBodies|align|bgColor|border|cellPadding|cellSpacing|frame|rules|summary|width|createTHead|deleteTHead|createTFoot|deleteTFoot|createCaption|deleteCaption|insertRow|deleteRow)|(content|closed|controllers|crypto|defaultStatus|directories|document|frameElement|frames|history|innerHeight|innerWidth|location|locationbar|menubar|name|navigator|opener|outerHeight|outerWidth|pageXOffset|pageYOffset|parent|personalbar|pkcs11|screen|availTop|availLeft|availHeight|availWidth|colorDepth|height|left|pixelDepth|top|width|scrollbars|scrollMaxX|scrollMaxY|scrollX|scrollY|self|sidebar|status|statusbar|toolbar|window|alert|atob|back|btoa|captureEvents|clearInterval|clearTimeout|close|confirm|dump|escape|find|forward|getAttention|getComputedStyle|getSelection|home|moveBy|moveTo|open|openDialog|print|prompt|releaseEvents|resizeBy|resizeTo|scroll|scrollBy|scrollByLines|scrollByPages|scrollTo|setInterval|setTimeout|sizeToContent|stop|unescape|updateCommands|onabort|onclose|ondragdrop|onerror|onload|onpaint|onreset|onscroll|onselect|onsubmit|onunload)|(XMLHttpRequest)|(length))\b|(\.(?:pop|push|reverse|shift|sort|splice|unshift|concat|join|slice)|(\.(?:getDate|getDay|getFullYear|getHours|getMilliseconds|getMinutes|getMonth|getSeconds|getTime|getTimezoneOffset|getUTCDate|getUTCDay|getUTCFullYear|getUTCHours|getUTCMilliseconds|getUTCMinutes|getUTCMonth|getUTCSeconds|setDate|setFullYear|setHours|setMilliseconds|setMinutes|setMonth|setSeconds|setTime|setUTCDate|setUTCFullYear|setUTCHours|setUTCMilliseconds|setUTCMinutes|setUTCMonth|setUTCSeconds|toDateString|toLocaleDateString|toLocaleTimeString|toTimeString|toUTCString))|(\.(?:apply|call))|(\.(?:toExponential|toFixed|toPrecision))|(\.(?:exec|test))|(\.(?:charAt|charCodeAt|concat|indexOf|lastIndexOf|localeCompare|match|replace|search|slice|split|substr|substring|toLocaleLowerCase|toLocaleUpperCase|toLowerCase|toUpperCase)))(\s*\(|$)/g; // collisions: bgColor, height, width, length, name
jush.links2.js_doc = /(^[ \t]*|\n\s*\*\s*|(?={))(@(?:augments|author|borrows|class|constant|constructor|constructs|default|deprecated|description|event|example|field|fileOverview|function|ignore|inner|lends|memberOf|name|namespace|param|private|property|public|requires|returns|see|since|static|throws|type|version)|(@argument)|(@extends)|(\{@link))(\b)/g;
/** Get callback for autocompletition
* @param string escaped empty identifier, e.g. `` for MySQL or [] for MS SQL
* @param Object<string, Array<string>> keys are table names, values are lists of columns
* @return Function see autocomplete()
*/
jush.autocompleteSql = function (esc, tablesColumns) {
	/**
	* key: regular expression; ' ' will be expanded to '\\s+', '\\w' to esc[0]+'?\\w'+esc[1]+'?', '$' will be appended
	* value: list of autocomplete words; '?' means to not use the word if it's already in the current query
	*/
	const keywordsDefault = {
		'^': ['SELECT', 'INSERT INTO', 'UPDATE', 'DELETE FROM', 'EXPLAIN'],
		'^EXPLAIN ': ['SELECT'],
		'^INSERT ': ['IGNORE'],
		'^INSERT .+\\) ': ['?VALUES', 'ON DUPLICATE KEY UPDATE'],
		'^UPDATE \\w+ ': ['SET'],
		'^UPDATE \\w+ SET .+ ': ['?WHERE'],
		'^DELETE FROM \\w+ ': ['WHERE'],
		' JOIN \\w+(( AS)? (?!(ON|USING|AS) )\\w+)? ': ['ON', 'USING'],
		'\\bSELECT ': ['*', 'DISTINCT'],
		'\\bSELECT .+ ': ['?FROM'],
		'\\bSELECT (?!.* (WHERE|GROUP BY|HAVING|ORDER BY|LIMIT) ).+ FROM .+ ': ['INNER JOIN', 'LEFT JOIN', '?WHERE'],
		'\\bSELECT (?!.* (HAVING|ORDER BY|LIMIT|OFFSET) ).+ FROM .+ ': ['?GROUP BY'],
		'\\bSELECT (?!.* (ORDER BY|LIMIT|OFFSET) ).+ FROM .+ ': ['?HAVING'],
		'\\bSELECT (?!.* (LIMIT|OFFSET) ).+ FROM .+ ': ['?ORDER BY'], // this matches prefixes without LIMIT|OFFSET and offers ORDER BY if it's not already used in prefix or suffix
		'\\bSELECT (?!.* (OFFSET) ).+ FROM .+ ': ['?LIMIT', '?OFFSET'],
		' ORDER BY (?!.* (LIMIT|OFFSET) ).+ ': ['DESC'],
	};
	
	/** Get list of strings for autocompletion
	* @param string
	* @param string
	* @param string
	* @return Object<string, number> keys are words, values are offsets
	*/
	function autocomplete(state, before, after) {
		if (/^(one|com|sql_apo|sqlite_apo)$/.test(state)) {
			return {};
		}
		before = before
			.replace(/\/\*.*?\*\/|\s--[^\n]*|'[^']+'/s, '') // strip comments and strings
			.replace(/.*;/s, '') // strip previous query
			.trimStart()
		;
		after = after.replace(/;.*/s, ''); // strip next query
		const query = before + after;
		const allTables = Object.keys(tablesColumns);
		const usedTables = findTables(query); // tables used by the current query
		const uniqueColumns = {};
		for (const table of Object.values(usedTables)) {
			for (const column of tablesColumns[table]) {
				uniqueColumns[column] = 0;
			}
		}
		const columns = Object.keys(uniqueColumns);
		if (columns.length > 50) {
			columns.length = 0;
		}
		if (Object.keys(usedTables).length > 1) {
			for (const alias in usedTables) {
				columns.push(alias + '.');
			}
		}
		
		const preferred = {
			'\\b(FROM|INTO|^UPDATE|JOIN) ': allTables, // all tables including the current ones (self-join)
			'\\b(^INSERT|USING) [^(]*\\(([^)]+, )?': columns, // offer columns right after '(' or after ','
			'(^UPDATE .+ SET| DUPLICATE KEY UPDATE| BY) (.+, )?': columns,
			' (WHERE|HAVING|AND|OR|ON|=) ': columns,
		};
		keywordsDefault['\\bSELECT( DISTINCT)? (?!.* FROM )(.+, )?'] = columns; // this is not in preferred because we prefer '*'
		
		const context = before.replace(escRe('[\\w`]+$'), ''); // in 'UPDATE tab.`co', context is 'UPDATE tab.'
		before = before.replace(escRe('.*[^\\w`]', 's'), ''); // in 'UPDATE tab.`co', before is '`co'
		
		const thisColumns = []; // columns in the current table ('table.')
		const match = context.match(escRe('`?(\\w+)`?\\.$'));
		if (match) {
			let table = match[1];
			if (!tablesColumns[table]) {
				table = usedTables[table];
			}
			if (tablesColumns[table]) {
				thisColumns.push(...tablesColumns[table]);
				preferred['\\.'] = thisColumns;
			}
		}

		if (query.includes(esc[0]) && !/^\w/.test(before)) { // if there's any ` in the query, use ` everywhere unless the user starts typing letters
			allTables.forEach(addEsc);
			columns.forEach(addEsc);
			thisColumns.forEach(addEsc);
		}
		
		const ac = {};
		for (const keywords of [preferred, keywordsDefault]) {
			for (const re in keywords) {
				if (context.match(escRe(re.replace(/ /g, '\\s+').replace(/\\w\+/g, '`?\\w+`?') + '$', 'is'))) {
					for (let keyword of keywords[re]) {
						if (keyword[0] == '?') {
							keyword = keyword.substring(1);
							if (query.match(new RegExp('\\s+' + keyword + '\\s+', 'i'))) {
								continue;
							}
						}
						if (keyword.length > before.length && keyword.toUpperCase().startsWith(before.toUpperCase())) {
							const isCol = (keywords[re] == columns || keywords[re] == thisColumns);
							ac[keyword + (isCol ? '' : ' ')] = before.length;
						}
					}
				}
			}
		}
		
		return ac;
	}
	
	function addEsc(val, key, array) {
		array[key] = esc[0] + val.replace(/\.?$/, esc[1] + '$&');
	}

	/** Change odd ` to esc[0], even to esc[1] */
	function escRe(re, flags) {
		let i = 0;
		return new RegExp(re.replace(/`/g, () => (esc[0] == '[' ? '\\' : '') + esc[i++ % 2]), flags);
	}

	/** @return Object<string, string> key is alias, value is actual table */
	function findTables(query) {
		const matches = query.matchAll(escRe('\\b(FROM|JOIN|INTO|UPDATE)\\s+(\\w+|`.+?`)((\\s+AS)?\\s+((?!(LEFT|INNER|JOIN|ON|USING|WHERE|GROUP|HAVING|ORDER|LIMIT)\\b)\\w+|`.+?`))?', 'gi')); //! handle `abc``def`
		const result = {};
		for (const match of matches) {
			const table = match[2].replace(escRe('^`|`$', 'g'), '');
			const alias = (match[5] ? match[5].replace(escRe('^`|`$', 'g'), '') : table);
			if (tablesColumns[table]) {
				result[alias] = table;
			}
		}
		if (!Object.keys(result).length) {
			for (const table in tablesColumns) {
				result[table] = table;
			}
		}
		return result;
	}

	// we open the autocomplete on word character, space, '(', '.' and '`'; textarea also triggers it on Backspace and Ctrl+Space
	autocomplete.openBy = escRe('^[\\w`(. ]$'); //! ignore . in 1.23

	return autocomplete;
};
jush.tr.pgsql = { sql_apo: /'/, sqlite_quo: /"/, pgsql_eot: /\$/, one: /--/, com_nest: /\/\*/, pgsql_pgsqlset: /(\b)(SHOW|SET)(\s+)/i, num: jush.num }; // standard_conforming_strings=off
jush.tr.pgsql_eot = { pgsql_eot2: /([a-z]\w*)?\$/i, _1: /()/ };
jush.tr.pgsql_eot2 = { }; // pgsql_eot2._2 to be set in pgsql_eot handler
jush.tr.pgsql_pgsqlset = { sql_apo: /'/, sqlite_quo: /"/, pgsql_eot: /\$/, one: /--/, com_nest: /\/\*/, num: jush.num, _1: /;|$/ };
jush.tr.pgsqlset = { _0: /$/ };

jush.urls.pgsql_pgsqlset = 'https://www.postgresql.org/docs/current/static/$key';
jush.urls.pgsql = ['https://www.postgresql.org/docs/current/static/$key',
	'sql-$1.html', 'sql-$1.html', 'sql-alteropclass.html', 'sql-createopclass.html', 'sql-dropopclass.html',
	'functions-datetime.html', 'functions-info.html', 'functions-logical.html', 'functions-comparison.html', 'functions-matching.html', 'functions-conditional.html', 'functions-subquery.html',
	'',
	'functions-math.html', 'functions-string.html', 'functions-binarystring.html', 'functions-formatting.html', 'functions-datetime.html', 'functions-geometry.html', 'functions-net.html', 'functions-sequence.html', 'functions-array.html', 'functions-aggregate.html', 'functions-srf.html', 'functions-info.html', 'functions-admin.html'
];
jush.urls.pgsqlset = ['https://www.postgresql.org/docs/current/static/runtime-config-$key.html#GUC-$1',
	'autovacuum', 'client', 'compatible', 'connection', 'custom', 'developer', 'file-locations', 'locks', 'logging', 'preset', 'query', 'resource', 'statistics', 'wal'
];

jush.links.pgsql_pgsqlset = { 'sql-$val.html': /.+/ };

jush.links2.pgsql = /(\b)(COMMIT\s+PREPARED|CREATE\s+ACCESS\s+METHOD|DROP\s+ACCESS\s+METHOD|DROP\s+OWNED|PREPARE\s+TRANSACTION|REASSIGN\s+OWNED|RELEASE\s+SAVEPOINT|ROLLBACK\s+PREPARED|SECURITY\s+LABEL|SET\s+CONSTRAINTS|SET\s+ROLE|SET\s+SESSION\s+AUTHORIZATION|SET\s+TRANSACTION|START\s+TRANSACTION|(ABORT|ALTER\s+AGGREGATE|ALTER\s+COLLATION|ALTER\s+CONVERSION|ALTER\s+DATABASE|ALTER\s+DEFAULT\s+PRIVILEGES|ALTER\s+DOMAIN|ALTER\s+EVENT\s+TRIGGER|ALTER\s+EXTENSION|ALTER\s+FOREIGN\s+DATA\s+WRAPPER|ALTER\s+FOREIGN\s+TABLE|ALTER\s+FUNCTION|ALTER\s+GROUP|ALTER\s+INDEX|ALTER\s+LANGUAGE|ALTER\s+LARGE\s+OBJECT|ALTER\s+MATERIALIZED\s+VIEW|ALTER\s+OPERATOR|ALTER\s+POLICY|ALTER\s+PROCEDURE|ALTER\s+PUBLICATION|ALTER\s+ROLE|ALTER\s+ROUTINE|ALTER\s+RULE|ALTER\s+SCHEMA|ALTER\s+SEQUENCE|ALTER\s+SERVER|ALTER\s+STATISTICS|ALTER\s+SUBSCRIPTION|ALTER\s+SYSTEM|ALTER\s+TABLE|ALTER\s+TABLESPACE|ALTER\s+TRIGGER|ALTER\s+TYPE|ALTER\s+USER(?:\s+MAPPING)?|ALTER\s+VIEW|ANALYZE|BEGIN|CALL|CHECKPOINT|CLOSE|CLUSTER|COMMENT|COMMIT|COPY|CREATE\s+AGGREGATE|CREATE\s+CAST|CREATE\s+COLLATION|CREATE\s+CONVERSION|CREATE\s+DATABASE|CREATE\s+DOMAIN|CREATE\s+EVENT\s+TRIGGER|CREATE\s+EXTENSION|CREATE\s+FOREIGN\s+DATA\s+WRAPPER|CREATE\s+FOREIGN\s+TABLE|CREATE\s+FUNCTION|CREATE\s+GROUP|CREATE\s+INDEX|CREATE\s+LANGUAGE|CREATE\s+MATERIALIZED\s+VIEW|CREATE\s+OPERATOR|CREATE\s+POLICY|CREATE\s+PROCEDURE|CREATE\s+PUBLICATION|CREATE\s+ROLE|CREATE\s+RULE|CREATE\s+SCHEMA|CREATE\s+SEQUENCE|CREATE\s+SERVER|CREATE\s+STATISTICS|CREATE\s+SUBSCRIPTION|CREATE\s+TABLE(?:\s+AS)?|CREATE\s+TABLESPACE|CREATE\s+TRANSFORM|CREATE\s+TRIGGER|CREATE\s+TYPE|CREATE\s+USER(?:\s+MAPPING)?|CREATE\s+VIEW|DEALLOCATE|DECLARE|DELETE|DISCARD|DO|DROP\s+AGGREGATE|DROP\s+CAST|DROP\s+COLLATION|DROP\s+CONVERSION|DROP\s+DATABASE|DROP\s+DOMAIN|DROP\s+EVENT\s+TRIGGER|DROP\s+EXTENSION|DROP\s+FOREIGN\s+DATA\s+WRAPPER|DROP\s+FOREIGN\s+TABLE|DROP\s+FUNCTION|DROP\s+GROUP|DROP\s+INDEX|DROP\s+LANGUAGE|DROP\s+MATERIALIZED\s+VIEW|DROP\s+OPERATOR|DROP\s+POLICY|DROP\s+PROCEDURE|DROP\s+PUBLICATION|DROP\s+ROLE|DROP\s+ROUTINE|DROP\s+RULE|DROP\s+SCHEMA|DROP\s+SEQUENCE|DROP\s+SERVER|DROP\s+STATISTICS|DROP\s+SUBSCRIPTION|DROP\s+TABLE|DROP\s+TABLESPACE|DROP\s+TRANSFORM|DROP\s+TRIGGER|DROP\s+TYPE|DROP\s+USER(?:\s+MAPPING)?|DROP\s+VIEW|END|EXECUTE|EXPLAIN|FETCH|GRANT|IMPORT\s+FOREIGN\s+SCHEMA|INSERT|LISTEN|LOAD|LOCK|MERGE|MOVE|NOTIFY|PREPARE|REFRESH\s+MATERIALIZED\s+VIEW|REINDEX|RESET|REVOKE|ROLLBACK(?:\s+TO)?|SAVEPOINT|SELECT|SELECT\s+INTO|SET|SHOW|TRUNCATE|UNLISTEN|UPDATE|VACUUM|VALUES)|(ALTER\s+OPERATOR\s+CLASS)|(CREATE\s+OPERATOR\s+CLASS)|(DROP\s+OPERATOR\s+CLASS)|(current_date|current_time|current_timestamp|localtime|localtimestamp|AT\s+TIME\s+ZONE)|(current_user|session_user|user)|(AND|NOT|OR)|(BETWEEN)|(LIKE|SIMILAR\s+TO)|(CASE|WHEN|THEN|ELSE|coalesce|nullif|greatest|least)|(EXISTS|IN|ANY|SOME|ALL)|(ANALYSE|ARRAY|AS|ASC|ASYMMETRIC|AUTHORIZATION|BINARY|BOTH|CAST|CHECK|COLLATE|COLLATION|COLUMN|CONCURRENTLY|CONSTRAINT|CREATE|CROSS|CURRENT_CATALOG|CURRENT_DATE|CURRENT_ROLE|CURRENT_SCHEMA|CURRENT_TIME|CURRENT_TIMESTAMP|CURRENT_USER|DEFAULT|DEFERRABLE|DESC|DISTINCT|EXCEPT|FALSE|FOR|FOREIGN|FREEZE|FROM|FULL|GROUP|HAVING|ILIKE|INITIALLY|INNER|INTERSECT|INTO|IS|ISNULL|JOIN|LATERAL|LEADING|LEFT|LIMIT|LOCALTIME|LOCALTIMESTAMP|NATURAL|NOTNULL|NULL|OFFSET|ON|ONLY|ORDER|OUTER|OVERLAPS|PLACING|PRIMARY|REFERENCES|RETURNING|RIGHT|SESSION_USER|SIMILAR|SYMMETRIC|SYSTEM_USER|TABLE|TABLESAMPLE|TO|TRAILING|TRUE|UNION|UNIQUE|USER|USING|VARIADIC|VERBOSE|WHERE|WINDOW|WITH))\b|\b(abs|cbrt|ceil|ceiling|degrees|exp|floor|ln|log|mod|pi|power|radians|random|round|setseed|sign|sqrt|trunc|width_bucket|acos|asin|atan|atan2|cos|cot|sin|tan|(bit_length|char_length|convert|lower|octet_length|overlay|position|substring|trim|upper|ascii|btrim|chr|decode|encode|initcap|length|lpad|ltrim|md5|pg_client_encoding|quote_ident|quote_literal|regexp_replace|repeat|replace|rpad|rtrim|split_part|strpos|substr|to_ascii|to_hex|translate)|(get_bit|get_byte|set_bit|set_byte|md5)|(to_char|to_date|to_number|to_timestamp)|(age|clock_timestamp|date_part|date_trunc|extract|isfinite|justify_days|justify_hours|justify_interval|now|statement_timestamp|timeofday|transaction_timestamp)|(area|center|diameter|height|isclosed|isopen|npoints|pclose|popen|radius|width|box|circle|lseg|path|point|polygon)|(abbrev|broadcast|family|host|hostmask|masklen|netmask|network|set_masklen|text|trunc)|(currval|nextval|setval)|(array_append|array_cat|array_dims|array_lower|array_prepend|array_to_string|array_upper|string_to_array)|(avg|bit_and|bit_or|bool_and|bool_or|count|every|max|min|sum|corr|covar_pop|covar_samp|regr_avgx|regr_avgy|regr_count|regr_intercept|regr_r2|regr_slope|regr_sxx|regr_sxy|regr_syy|stddev|stddev_pop|stddev_samp|variance|var_pop|var_samp)|(generate_series)|(current_database|current_schema|current_schemas|inet_client_addr|inet_client_port|inet_server_addr|inet_server_port|pg_my_temp_schema|pg_is_other_temp_schema|pg_postmaster_start_time|version|has_database_privilege|has_function_privilege|has_language_privilege|has_schema_privilege|has_table_privilege|has_tablespace_privilege|pg_has_role|pg_conversion_is_visible|pg_function_is_visible|pg_operator_is_visible|pg_opclass_is_visible|pg_table_is_visible|pg_type_is_visible|format_type|pg_get_constraintdef|pg_get_expr|pg_get_indexdef|pg_get_ruledef|pg_get_serial_sequence|pg_get_triggerdef|pg_get_userbyid|pg_get_viewdef|pg_tablespace_databases|col_description|obj_description|shobj_description)|(current_setting|set_config|pg_cancel_backend|pg_reload_conf|pg_rotate_logfile|pg_start_backup|pg_stop_backup|pg_switch_xlog|pg_current_xlog_location|pg_current_xlog_insert_location|pg_xlogfile_name_offset|pg_xlogfile_name|pg_column_size|pg_database_size|pg_relation_size|pg_size_pretty|pg_tablespace_size|pg_total_relation_size|pg_ls_dir|pg_read_file|pg_stat_file|pg_advisory_lock|pg_advisory_lock_shared|pg_try_advisory_lock|pg_try_advisory_lock_shared|pg_advisory_unlock|pg_advisory_unlock_shared|pg_advisory_unlock_all))(\s*\(|$)/gi; // collisions: IN, ANY, SOME, ALL (array), trunc, md5, abbrev
jush.links2.pgsqlset = /(\b)(autovacuum|log_autovacuum_min_duration|autovacuum_max_workers|autovacuum_naptime|autovacuum_vacuum_threshold|autovacuum_analyze_threshold|autovacuum_vacuum_scale_factor|autovacuum_analyze_scale_factor|autovacuum_freeze_max_age|autovacuum_vacuum_cost_delay|autovacuum_vacuum_cost_limit|(search_path|default_tablespace|temp_tablespaces|check_function_bodies|default_transaction_isolation|default_transaction_read_only|session_replication_role|statement_timeout|vacuum_freeze_table_age|vacuum_freeze_min_age|xmlbinary|xmloption|datestyle|intervalstyle|timezone|timezone_abbreviations|extra_float_digits|client_encoding|lc_messages|lc_monetary|lc_numeric|lc_time|default_text_search_config|dynamic_library_path|gin_fuzzy_search_limit|local_preload_libraries)|(add_missing_from|array_nulls|backslash_quote|default_with_oids|escape_string_warning|regex_flavor|sql_inheritance|standard_conforming_strings|synchronize_seqscans|transform_null_equals)|(listen_addresses|port|max_connections|superuser_reserved_connections|unix_socket_directory|unix_socket_group|unix_socket_permissions|bonjour_name|tcp_keepalives_idle|tcp_keepalives_interval|tcp_keepalives_count|authentication_timeout|ssl|ssl_renegotiation_limit|ssl_ciphers|password_encryption|krb_server_keyfile|krb_srvname|krb_caseins_users|db_user_namespace)|(custom_variable_classes)|(allow_system_table_mods|debug_assertions|ignore_system_indexes|post_auth_delay|pre_auth_delay|trace_notify|trace_sort|wal_debug|zero_damaged_pages)|(data_directory|config_file|hba_file|ident_file|external_pid_file)|(deadlock_timeout|max_locks_per_transaction)|(log_destination|logging_collector|log_directory|log_filename|log_rotation_age|log_rotation_size|log_truncate_on_rotation|syslog_facility|syslog_ident|silent_mode|client_min_messages|log_min_messages|log_error_verbosity|log_min_error_statement|log_min_duration_statement|log_checkpoints|log_connections|log_disconnections|log_duration|log_hostname|log_line_prefix|log_lock_waits|log_statement|log_temp_files|log_timezone)|(block_size|integer_datetimes|lc_collate|lc_ctype|max_function_args|max_identifier_length|max_index_keys|segment_size|server_encoding|server_version|server_version_num|wal_block_size|wal_segment_size)|(enable_bitmapscan|enable_hashagg|enable_hashjoin|enable_indexscan|enable_mergejoin|enable_nestloop|enable_seqscan|enable_sort|enable_tidscan|seq_page_cost|random_page_cost|cpu_tuple_cost|cpu_index_tuple_cost|cpu_operator_cost|effective_cache_size|geqo|geqo_threshold|geqo_effort|geqo_pool_size|geqo_generations|geqo_selection_bias|default_statistics_target|constraint_exclusion|cursor_tuple_fraction|from_collapse_limit|join_collapse_limit)|(shared_buffers|temp_buffers|max_prepared_transactions|work_mem|maintenance_work_mem|max_stack_depth|max_files_per_process|shared_preload_libraries|vacuum_cost_delay|vacuum_cost_page_hit|vacuum_cost_page_miss|vacuum_cost_page_dirty|vacuum_cost_limit|bgwriter_delay|bgwriter_lru_maxpages|bgwriter_lru_multiplier|effective_io_concurrency)|(track_activities|track_activity_query_size|track_counts|track_functions|update_process_title|stats_temp_directory)|(fsync|synchronous_commit|wal_sync_method|full_page_writes|wal_buffers|wal_writer_delay|commit_delay|commit_siblings|checkpoint_segments|checkpoint_timeout|checkpoint_completion_target|checkpoint_warning|archive_mode|archive_command|archive_timeout))(\b)/gi;
jush.tr.txt = { php: jush.php };

jush.tr.mssql = { sqlite_apo: /'/, sqlite_quo: /"/, one: /--/, com: /\/\*/, mssql_bra: /\[/, num: jush.num }; // QUOTED IDENTIFIER = OFF
jush.tr.mssql_bra = { _0: /]]/, _1: /]/ };

jush.urls.mssql = ['https://msdn.microsoft.com/library/$key.aspx',
	'ms181700', 'ms178543', 'ms188372', 'ms189526', 'ms186865', 'ms178578', 'ms174387', 'ms190337', 'ms190487', 'ms187804', 'ms187377', 'ms188386', 'ms188929', 'ms187922', 'ms188362', 'ms177603', 'ms181271', 'ms188365', 'ms181765', 'ms187368', 'ms176089', 'ms188748', 'ms175035', 'ms188387', 'ms177938', 'ms184391', 'ms178628', 'ms190295', 'ms181708', 'ms174366', 'bb630352', 'ms187819', 'bb677335', 'bb630289', 'ms188782', 'ms187746', 'ms188927', 'ms180169', 'ms189835', 'ms188338', 'ms189748', 'ms182587', 'ms182706', 'ms176030', 'ms177521', 'ms188055', 'ms188332', 'ms181362', 'ms188336', 'ms180152', 'ms173773', 'ms173812', 'ms177634', 'cc280766', 'cc280487', 'ms178624', 'ms188037', 'ms180188', 'ms187965', 'ms177673', 'ms180199', 'bb677290', 'ms186775', 'ms182717', 'ms177682', 'ms174335', 'ms187745', 'ms188029', 'ms188795', 'ms173730', 'ms186764', 'ms180016', 'ms179859', 'bb510625', 'ms179882', 'ms174987', 'ms186939', 'ms189455', 'ms187993', 'ms190500', 'ms174433', 'ms190499', 'ms190322', 'ms188361', 'ms188385', 'ms177564', 'ms189461', 'ms176047', 'ms190372', 'ms186336', 'ms187972', 'ms174998', 'ms178632', 'ms187728', 'ms181299', 'ms174973', 'ms182776', 'ms188378', 'ms189499', 'ms188407', 'ms190356', 'ms188767', 'ms182418', 'ms175064', 'ms173829', 'bb677243', 'ms189463', 'ms175976', 'ms177570', 'ms180026', 'ms187942', 'ms177523', 'ms187348', 'ms189466', 'ms188366', 'ms186290', 'ms187331', 'ms188047', 'ms178642', 'ms175972', 'ms177607', 'ms186838', 'ms189797',
	'ms182741', 'ms181491', 'ms189524', 'ms174430', 'bb934170', 'ms187798', 'ms178528', 'ms189522', 'bb677184', 'ms176061', 'cc280404', 'bb677241', 'ms173565', 'ms181591', 'ms189453', 'bb677289', 'ms189520', 'ms187317', 'cc280405', 'ms186755', 'ms188783', 'ms189751', 'ms174382', 'ms187744', 'ms187802', 'ms179854', 'ms187926', 'ms190495', 'ms178024', 'bb895329', 'ms187936', 'ms186742', 'ms188064', 'ms189462', 'cc280448', 'cc280767', 'ms190332', 'ms188038', 'ms188357', 'ms177544', 'ms174979', 'ms189799', 'ms175007', 'ms173463', 'ms187956', 'bb934146', 'ms176009',
	'ms186847', 'ms177517', 'ms177514', 'ms188389', 'bb964728', 'ms179906', 'ms190475', 'ms189450', 'bb677309', 'ms178613', 'cc280479', 'bb630256', 'ms188747', 'ms181586', 'ms174414', 'bb630257', 'ms188403', 'ms184393', 'cc280482', 'ms190290', 'ms176118', 'ms188012', 'ms180071', 'ms186728', 'ms187759', 'ms181249', 'ms174969', 'ms190480', 'ms177539', 'bb933779', 'ms174988', 'ms189449', 'ms186791', 'ms186751', 'cc280899', 'cc280603', 'ms174990', 'ms186977', 'ms175075', 'ms182698', 'ms174996', 'ms173790', 'ms173497', 'ms174407', 'ms189438', 'ms173492', 'bb933867', 'ms189448',
	'ms188900', 'ms186711', 'ms187311', 'ms187359', 'bb933778', 'ms189511', 'ms187923', 'bb677336', 'ms174269', 'cc280645', 'bb630389', 'ms186332', 'bb630368', 'ms176095', 'ms188359', 'cc280871', 'ms186967', 'ms188388', 'ms189828', 'ms186937', 'ms187345', 'ms186307', 'ms190347', 'ms189762', 'ms189529', 'ms190363', 'bb934013', 'bb934024', 'ms189775', 'ms187353', 'ms173423', 'cc280563', 'cc280682', 'ms176036', 'ms187788', 'ms189440', 'ms190273', 'ms176072', 'ms176060', 'ms173846', 'bb895361', 'ms189778',
	'ms189800', 'ms178627', 'ms189749', 'ms178647', 'ms189770', 'ms177545', 'ms181581', 'ms188750', 'ms189508', 'cc627393', 'ms181746', 'ms173854', 'ms177677', 'ms173784', 'ms187928', 'ms189818', 'ms187324', 'ms180142', 'ms187323', 'ms186323', 'ms189788', 'ms188920', 'ms190349', 'ms190305', 'ms188732', 'ms174974', 'ms174968', 'ms186329', 'bb895240', 'ms187787', 'ms189760', 'ms180125', 'ms189534', 'ms188919', 'ms188921', 'ms175997', 'ms190317', 'cc627408', 'ms187352', 'ms188751', 'ms176050', 'ms177609', 'ms187319', 'ms176049', 'ms186823', 'ms173486', 'ms186819', 'ms189794', 'ms174395', 'ms174420', 'ms176052', 'ms186274', 'ms189753', 'ms188796', 'ms189507', 'ms178601', 'ms181860', 'ms365420', 'ms182559', 'ms188910', 'ms178566', 'ms173825', 'ms188753', 'ms186950', 'ms188061', 'ms174361', 'ms190357', 'ms178600', 'ms190358', 'ms175069', 'ms188398', 'ms178567', 'ms180031', 'ms173781', 'ms179857', 'ms182063', 'ms186275', 'ms181399', 'ms186980', 'ms176088', 'ms188069', 'ms188401', 'ms178531', 'ms186788', 'ms176078', 'ms177652', 'ms190370', 'ms188418', 'bb934014', 'ms181825', 'ms174960', 'ms188383', 'ms178635', 'ms178544', 'bb510624', 'ms187718', 'ms189802', 'ms174415', 'ms177605', 'ms178598', 'ms175098', 'ms189795', 'ms189834', 'ms186773', 'ms187729', 'ms178545', 'ms186271', 'cc627401', 'ms176015', 'ms187347', 'ms184325', 'ms186272', 'ms187385', 'ms189457', 'cc645960', 'ms177601', 'ms190329', 'ms190319', 'ms175121', 'ms345412', 'ms174400', 'ms177827', 'ms187751', 'ms179916', 'bb839514', 'ms187813', 'ms182673', 'ms190348', 'ms189786', 'ms175126', 'ms177562', 'ms176090', 'ms190328', 'ms186301', 'bb326599', 'ms176105', 'ms188390', 'ms179856', 'ms188427', 'ms190312', 'ms186918', 'bb895328', 'ms189492', 'ms188006', 'bb895239', 'ms188395', 'ms186915', 'ms189512', 'ms174276', 'ms189830', 'dd822792', 'dd822791', 'ms176114', 'ms189742', 'ms178592', 'ms177610', 'ms176102', 'ms187365', 'ms186963', 'ms176069', 'ms186862', 'ms174383', 'ms180040', 'ms177532', 'ms175003', 'ms186734', 'ms181406', 'ms178660', 'ms188797', 'ms175068', 'ms190315', 'ms174396', 'ms177587', 'ms175001', 'ms186297', 'ms188420', 'ms187913', 'ms187920', 'ms188377', 'ms187384', 'ms187950', 'ms178550', 'ms176108', 'ms173569', 'ms190330', 'ms190474', 'ms176080', 'ms189527', 'ms188043', 'ms187748', 'ms187810', 'ms176042', 'ms187934', 'ms179889', 'ms174427', 'bb677244', 'bb630353', 'bb677334', 'cc645882', 'bb630387', 'ms179930', 'ms190338', 'ms186881', 'ms176068', 'ms187362', 'bb630335', 'ms182737', 'ms181628', 'ms189750', 'ms188419', 'ms180059', 'ms187326', 'ms180055', 'ms186738', 'ms181466', 'ms188014', 'ms188735', 'ms178631', 'ms187791', 'ms187339', 'ms190316', 'ms186313'
];

jush.links2.mssql = /(\b)(ADD(?:\s+COUNTER)?\s+SIGNATURE|(ALL)|(AND)|(ANY)|(BACKUP)|(BACKUP\s+CERTIFICATE)|(BACKUP\s+MASTER\s+KEY)|(BACKUP\s+SERVICE\s+MASTER\s+KEY)|(BEGIN)|(BEGIN\s+CONVERSATION\s+TIMER)|(BEGIN\s+DIALOG)|(BEGIN\s+DISTRIBUTED\s+(?:TRANSACTION|TRAN))|(BEGIN\s+(?:TRANSACTION|TRAN))|(BETWEEN)|((?:var)?binary)|(bit)|(BREAK)|(BULK\s+INSERT)|(CASE)|(CATCH)|((?:var)?char)|(CHECKPOINT)|(CLOSE)|(CLOSE\s+MASTER\s+KEY)|(CLOSE\s+(?:SYMMETRIC\s+KEY|ALL\s+SYMMETRIC\s+KEYS))|(COLLATE)|(COMMIT)|(COMMIT\s+(?:TRANSACTION|TRAN))|(COMPUTE)|(CONTINUE)|(date)|(datetime)|(datetime2)|(datetimeoffset)|(DEALLOCATE)|(decimal|numeric)|(DECLARE)|(DECLARE\s+CURSOR)|(DELETE)|(DENY)|(DISABLE\s+TRIGGER)|(ELSE)|(ENABLE\s+TRIGGER)|(END)|(END\s+CONVERSATION)|(EXCEPT|INTERSECT)|(EXECUTE|EXEC)|((?:EXECUTE|EXEC)\s+AS)|(EXISTS)|(FETCH)|(float|real)|(FOR)|(FROM)|(geography)|(geometry)|(GET\s+CONVERSATION\s+GROUP)|(GO)|(GOTO)|(GRANT)|(GROUP\s+BY)|(HAVING)|(hierarchyid)|(IDENTITY)|(IF)|(IN)|(INSERT)|((?:big|small|tiny)?int)|(INTO)|(IS(?:\s+NOT)?\s+NULL)|(KILL)|(KILL\s+QUERY\s+NOTIFICATION\s+SUBSCRIPTION)|(KILL\s+STATS\s+JOB)|(LIKE)|(MERGE)|((?:small)?money)|(MOVE\s+CONVERSATION)|(nchar|nvarchar)|(NOT)|(ntext|text|image)|(OPEN)|(OPEN\s+MASTER\s+KEY)|(OPEN\s+SYMMETRIC\s+KEY)|(OPTION)|(OR)|(ORDER\s+BY)|(OUTPUT)|(OVER)|(PRINT)|(RESTORE)|(RESTORE\s+MASTER\s+KEY)|(RESTORE\s+SERVICE\s+MASTER\s+KEY)|(RETURN)|(REVERT)|(REVOKE)|(ROLLBACK\s+TRANSACTION)|(ROLLBACK\s+WORK)|(rowversion)|(SAVE\s+TRANSACTION)|(SELECT)|(SEND)|(SET)|(SHUTDOWN)|(smalldatetime)|(SOME|ANY)|(sql_variant)|(time)|(TOP)|(TRY)|(TRUNCATE\s+TABLE)|(UNION)|(uniqueidentifier)|(UPDATE)|(UPDATE\s+STATISTICS)|(UPDATETEXT)|(USE)|(VAR)|(WAITFOR)|(WHERE)|(WHILE)|(WITH)|(WITH\s+XMLNAMESPACES)|(WRITETEXT)|(XACT_STATE)|(CREATE\s+AGGREGATE)|(CREATE\s+APPLICATION\s+ROLE)|(CREATE\s+ASSEMBLY)|(CREATE\s+ASYMMETRIC\s+KEY)|(CREATE\s+BROKER\s+PRIORITY)|(CREATE\s+CERTIFICATE)|(CREATE\s+CONTRACT)|(CREATE\s+CREDENTIAL)|(CREATE\s+CRYPTOGRAPHIC\s+PROVIDER)|(CREATE\s+DATABASE)|(CREATE\s+DATABASE\s+AUDIT\s+SPECIFICATION)|(CREATE\s+DATABASE\s+ENCRYPTION\s+KEY)|(CREATE\s+DEFAULT)|(CREATE\s+ENDPOINT)|(CREATE\s+EVENT\s+NOTIFICATION)|(CREATE\s+EVENT\s+SESSION)|(CREATE\s+FULLTEXT\s+CATALOG)|(CREATE\s+FULLTEXT\s+INDEX)|(CREATE\s+FULLTEXT\s+STOPLIST)|(CREATE\s+FUNCTION)|(CREATE(?:\s+UNIQUE)?\s+INDEX)|(CREATE\s+LOGIN)|(CREATE\s+MASTER\s+KEY)|(CREATE\s+MESSAGE\s+TYPE)|(CREATE\s+PARTITION\s+FUNCTION)|(CREATE\s+PARTITION\s+SCHEME)|(CREATE\s+PROCEDURE)|(CREATE\s+QUEUE)|(CREATE\s+REMOTE\s+SERVICE\s+BINDING)|(CREATE\s+RESOURCE\s+POOL)|(CREATE\s+ROLE)|(CREATE\s+ROUTE)|(CREATE\s+RULE)|(CREATE\s+SCHEMA)|(CREATE\s+SERVER\s+AUDIT)|(CREATE\s+SERVER\s+AUDIT\s+SPECIFICATION)|(CREATE\s+SERVICE)|(CREATE\s+STATISTICS)|(CREATE\s+SYMMETRIC\s+KEY)|(CREATE\s+SYNONYM)|(CREATE\s+TABLE)|(CREATE\s+TRIGGER)|(CREATE\s+TYPE)|(CREATE\s+USER)|(CREATE\s+VIEW)|(CREATE\s+WORKLOAD\s+GROUP)|(CREATE\s+XML\s+SCHEMA\s+COLLECTION)|(DROP\s+AGGREGATE)|(DROP\s+APPLICATION\s+ROLE)|(DROP\s+ASSEMBLY)|(DROP\s+ASYMMETRIC\s+KEY)|(DROP\s+BROKER\s+PRIORITY)|(DROP\s+CERTIFICATE)|(DROP\s+CONTRACT)|(DROP\s+CREDENTIAL)|(DROP\s+CRYPTOGRAPHIC\s+PROVIDER)|(DROP\s+DATABASE)|(DROP\s+DATABASE\s+AUDIT\s+SPECIFICATION)|(DROP\s+DATABASE\s+ENCRYPTION\s+KEY)|(DROP\s+DEFAULT)|(DROP\s+ENDPOINT)|(DROP\s+EVENT\s+NOTIFICATION)|(DROP\s+EVENT\s+SESSION)|(DROP\s+FULLTEXT\s+CATALOG)|(DROP\s+FULLTEXT\s+INDEX)|(DROP\s+FULLTEXT\s+STOPLIST)|(DROP\s+FUNCTION)|(DROP\s+INDEX)|(DROP\s+LOGIN)|(DROP\s+MASTER\s+KEY)|(DROP\s+MESSAGE\s+TYPE)|(DROP\s+PARTITION\s+FUNCTION)|(DROP\s+PARTITION\s+SCHEME)|(DROP\s+PROCEDURE)|(DROP\s+QUEUE)|(DROP\s+REMOTE\s+SERVICE\s+BINDING)|(DROP\s+RESOURCE\s+POOL)|(DROP\s+ROLE)|(DROP\s+ROUTE)|(DROP\s+RULE)|(DROP\s+SCHEMA)|(DROP\s+SERVER\s+AUDIT)|(DROP\s+SERVER\s+AUDIT\s+SPECIFICATION)|(DROP\s+SERVICE)|(DROP\s+SIGNATURE)|(DROP\s+STATISTICS)|(DROP\s+SYMMETRIC\s+KEY)|(DROP\s+SYNONYM)|(DROP\s+TABLE)|(DROP\s+TRIGGER)|(DROP\s+TYPE)|(DROP\s+USER)|(DROP\s+VIEW)|(DROP\s+WORKLOAD\s+GROUP)|(DROP\s+XML\s+SCHEMA\s+COLLECTION)|(ALTER\s+APPLICATION\s+ROLE)|(ALTER\s+ASSEMBLY)|(ALTER\s+ASYMMETRIC\s+KEY)|(ALTER\s+AUTHORIZATION)|(ALTER\s+BROKER\s+PRIORITY)|(ALTER\s+CERTIFICATE)|(ALTER\s+CREDENTIAL)|(ALTER\s+CRYPTOGRAPHIC\s+PROVIDER)|(ALTER\s+DATABASE)|(ALTER\s+DATABASE\s+AUDIT\s+SPECIFICATION)|(ALTER\s+DATABASE\s+ENCRYPTION\s+KEY)|(ALTER\s+ENDPOINT)|(ALTER\s+EVENT\s+SESSION)|(ALTER\s+FULLTEXT\s+CATALOG)|(ALTER\s+FULLTEXT\s+INDEX)|(ALTER\s+FULLTEXT\s+STOPLIST)|(ALTER\s+FUNCTION)|(ALTER\s+INDEX)|(ALTER\s+LOGIN)|(ALTER\s+MASTER\s+KEY)|(ALTER\s+MESSAGE\s+TYPE)|(ALTER\s+PARTITION\s+FUNCTION)|(ALTER\s+PARTITION\s+SCHEME)|(ALTER\s+PROCEDURE)|(ALTER\s+QUEUE)|(ALTER\s+REMOTE\s+SERVICE\s+BINDING)|(ALTER\s+RESOURCE\s+GOVERNOR)|(ALTER\s+RESOURCE\s+POOL)|(ALTER\s+ROLE)|(ALTER\s+ROUTE)|(ALTER\s+SCHEMA)|(ALTER\s+SERVER\s+AUDIT)|(ALTER\s+SERVER\s+AUDIT\s+SPECIFICATION)|(ALTER\s+SERVICE)|(ALTER\s+SERVICE\s+MASTER\s+KEY)|(ALTER\s+SYMMETRIC\s+KEY)|(ALTER\s+TABLE)|(ALTER\s+TRIGGER)|(ALTER\s+USER)|(ALTER\s+VIEW)|(ALTER\s+WORKLOAD\s+GROUP)|(ALTER\s+XML\s+SCHEMA\s+COLLECTION))\b|\b(ABS|(ACOS)|(APPLOCK_MODE)|(APPLOCK_TEST)|(APP_NAME)|(ASCII)|(ASIN)|(ASSEMBLYPROPERTY)|(ASYMKEY_ID)|(ASYMKEYPROPERTY)|(ATAN)|(ATN2)|(AVG)|(BINARY_CHECKSUM )|(CAST|CONVERT)|(CEILING)|(CertProperty)|(Cert_ID)|(CHAR)|(CHARINDEX)|(CHECKSUM)|(CHECKSUM_AGG)|(COALESCE)|(COLLATIONPROPERTY)|(COL_LENGTH)|(COL_NAME)|(COLUMNPROPERTY)|(COLUMNS_UPDATED)|(CONNECTIONPROPERTY)|(CONTAINS)|(CONTAINSTABLE)|(CONTEXT_INFO)|(CONVERT)|(COS)|(COT)|(COUNT)|(COUNT_BIG)|(CRYPT_GEN_RANDOM)|(CURRENT_REQUEST_ID)|(CURRENT_TIMESTAMP)|(CURRENT_USER)|(CURSOR_STATUS)|(DATABASE_PRINCIPAL_ID)|(DATABASEPROPERTY)|(DATABASEPROPERTYEX)|(DATALENGTH)|(DATEADD)|(DATEDIFF)|(DATENAME)|(DATEPART)|(DAY)|(DB_ID)|(DB_NAME)|(DBCC)|(DECRYPTBYASYMKEY)|(DECRYPTBYCERT)|(DECRYPTBYKEY)|(DECRYPTBYKEYAUTOASYMKEY)|(DECRYPTBYKEYAUTOCERT)|(DECRYPTBYPASSPHRASE)|(DEGREES)|(DENSE_RANK)|(DIFFERENCE)|(ENCRYPTBYASYMKEY)|(ENCRYPTBYCERT)|(ENCRYPTBYKEY)|(ENCRYPTBYPASSPHRASE)|(ERROR_LINE)|(ERROR_MESSAGE)|(ERROR_NUMBER)|(ERROR_PROCEDURE)|(ERROR_SEVERITY)|(ERROR_STATE)|(EVENTDATA)|(EXP)|(FILE_ID)|(FILE_IDEX)|(FILE_NAME)|(FILEGROUP_ID)|(FILEGROUP_NAME)|(FILEGROUPPROPERTY)|(FILEPROPERTY)|(FLOOR)|(FORMATMESSAGE)|(FREETEXT)|(FREETEXTTABLE)|(FULLTEXTCATALOGPROPERTY)|(FULLTEXTSERVICEPROPERTY)|(GET_FILESTREAM_TRANSACTION_CONTEXT)|(GET_TRANSMISSION_STATUS)|(GETANSINULL)|(GETDATE)|(GETUTCDATE)|(GROUPING)|(GROUPING_ID)|(HAS_DBACCESS)|(HAS_PERMS_BY_NAME)|(HASHBYTES)|(HOST_ID)|(HOST_NAME)|(IDENT_CURRENT)|(IDENT_INCR)|(IDENT_SEED)|(INDEXKEY_PROPERTY)|(INDEXPROPERTY)|(INDEX_COL)|(IS_MEMBER)|(IS_OBJECTSIGNED)|(IS_SRVROLEMEMBER)|(ISDATE)|(ISNULL)|(ISNUMERIC)|(Key_GUID)|(Key_ID)|(KEY_NAME)|(LEFT)|(LEN)|(LOG)|(LOG10)|(LOGINPROPERTY)|(LOWER)|(LTRIM)|(MAX)|(MIN)|(MIN_ACTIVE_ROWVERSION)|(MONTH)|(NCHAR)|(NEWID)|(NEWSEQUENTIALID)|(NTILE)|(NULLIF)|(OBJECT_DEFINITION)|(OBJECT_ID)|(OBJECT_NAME)|(OBJECT_SCHEMA_NAME)|(OBJECTPROPERTY)|(OBJECTPROPERTYEX)|(OPENDATASOURCE)|(OPENQUERY)|(OPENROWSET)|(OPENXML)|(ORIGINAL_DB_NAME)|(ORIGINAL_LOGIN)|(PARSENAME)|(PathName)|(PATINDEX)|(PERMISSIONS)|(PI)|(POWER)|(PUBLISHINGSERVERNAME)|(PWDCOMPARE)|(PWDENCRYPT)|(QUOTENAME)|(RADIANS)|(RAISERROR)|(RAND)|(RANK)|(READTEXT)|(RECEIVE)|(RECONFIGURE)|(REPLACE)|(REPLICATE)|(REVERSE)|(RIGHT)|(ROUND)|(ROW_NUMBER)|(ROWCOUNT_BIG)|(RTRIM)|(SCHEMA_ID)|(SCHEMA_NAME)|(SCOPE_IDENTITY)|(SERVERPROPERTY)|(SESSION_USER)|(SESSIONPROPERTY)|(SETUSER)|(SIGN)|(SignByAsymKey)|(SignByCert)|(SIN)|(SOUNDEX)|(SPACE)|(SQL_VARIANT_PROPERTY)|(SQRT)|(SQUARE)|(STATS_DATE)|(STDEV)|(STDEVP)|(STR)|(STUFF)|(SUBSTRING)|(SUM)|(SUSER_ID)|(SUSER_NAME)|(SUSER_SID)|(SUSER_SNAME)|(SWITCHOFFSET)|(SYSDATETIME)|(SYSDATETIMEOFFSET)|(SYMKEYPROPERTY)|(SYSUTCDATETIME)|(SYSTEM_USER)|(TAN)|(TERTIARY_WEIGHTS)|(TEXTPTR)|(TEXTVALID)|(TODATETIMEOFFSET)|(TRIGGER_NESTLEVEL)|(TYPE_ID)|(TYPE_NAME)|(TYPEPROPERTY)|(UNICODE)|(UPDATE)|(UPPER)|(USER)|(USER_ID)|(USER_NAME)|(VARP)|(VerifySignedByCert)|(VerifySignedByAsymKey)|(xml)|(xml_schema_namespace)|(YEAR))(\s*\(|$)/gi; // collisions: IDENTITY
jush.tr.sql = { one: /-- |#|--(?=\n|$)/, com_code: /\/\*![0-9]*|\*\//, com: /\/\*/, sql_sqlset: /(\s*)(SET)(\s+|$)(?!NAMES\b|CHARACTER\b|PASSWORD\b|(?:GLOBAL\s+|SESSION\s+)?TRANSACTION\b|@[^@]|NEW\.|OLD\.)/i, sql_code: /()/ };
jush.tr.sql_code = { sql_apo: /'/, sql_quo: /"/, bac: /`/, one: /-- |#|--(?=\n|$)/, com_code: /\/\*![0-9]*|\*\//, com: /\/\*/, sql_var: /\B@/, num: jush.num, _1: /;|\b(THEN|ELSE|LOOP|REPEAT|DO)\b/i };
jush.tr.sql_sqlset = { one: /-- |#|--(?=\n|$)/, com: /\/\*/, sqlset_val: /=/, _1: /;|$/ };
jush.tr.sqlset_val = { sql_apo: /'/, sql_quo: /"/, bac: /`/, one: /-- |#|--(?=\n|$)/, com: /\/\*/, _1: /,/, _2: /;|$/, num: jush.num }; //! comma can be inside function call
jush.tr.sqlset = { _0: /$/ }; //! jump from SHOW VARIABLES LIKE ''
jush.tr.sqlstatus = { _0: /$/ }; //! jump from SHOW STATUS LIKE ''
jush.tr.com_code = { _1: /()/ };

jush.urls.sql_sqlset = 'https://dev.mysql.com/doc/mysql/en/$key';
jush.urls.sql = ['https://dev.mysql.com/doc/mysql/en/$key',
	'alter-event.html', 'alter-table.html', 'alter-view.html', 'analyze-table.html', 'create-event.html', 'create-function.html', 'create-procedure.html', 'create-index.html', 'create-table.html', 'create-trigger.html', 'create-view.html', 'drop-index.html', 'drop-table.html', 'begin-end.html', 'optimize-table.html', 'repair-table.html', 'set-transaction.html', 'show-columns.html', 'show-engines.html', 'show-index.html', 'show-processlist.html', 'show-status.html', 'show-tables.html', 'show-variables.html',
	'$1.html', '$1-statement.html', 'if-statement.html', 'repeat-statement.html', 'truncate-table.html', 'commit.html', 'savepoints.html', 'lock-tables.html', 'charset-connection.html', 'insert-on-duplicate.html', 'fulltext-search.html', 'example-auto-increment.html',
	'comparison-operators.html#operator_$1', 'comparison-operators.html#function_$1', 'any-in-some-subqueries.html', 'all-subqueries.html', 'exists-and-not-exists-subqueries.html', 'group-by-modifiers.html', 'string-functions.html#operator_$1', 'string-comparison-functions.html#operator_$1', 'regexp.html#operator_$1', 'regexp.html#operator_regexp', 'logical-operators.html#operator_$1', 'control-flow-functions.html#operator_$1', 'arithmetic-functions.html#operator_$1', 'cast-functions.html#operator_$1', 'date-and-time-functions.html#function_$1', 'date-and-time-functions.html#function_date-add',
	'', // keywords without link
	'numeric-type-syntax.html', 'date-and-time-type-syntax.html', 'string-type-syntax.html', 'mysql-spatial-datatypes.html',
	'mathematical-functions.html#function_$1', 'information-functions.html#function_$1',
	'$1-storage-engine.html', 'merge-storage-engine.html',
	'partitioning-range.html', 'partitioning-list.html', 'partitioning-columns.html', 'partitioning-hash.html', 'partitioning-linear-hash.html', 'partitioning-key.html',
	'comparison-operators.html#function_$1', 'control-flow-functions.html#function_$1', 'string-functions.html#function_$1', 'string-comparison-functions.html#function_$1', 'mathematical-functions.html#function_$1', 'date-and-time-functions.html#function_$1', 'cast-functions.html#function_$1', 'xml-functions.html#function_$1', 'bit-functions.html#function_$1', 'encryption-functions.html#function_$1', 'information-functions.html#function_$1', 'miscellaneous-functions.html#function_$1', 'group-by-functions.html#function_$1',
	'functions-to-convert-geometries-between-formats.html#function_asbinary',
	'functions-to-convert-geometries-between-formats.html#function_astext',
	'functions-for-testing-spatial-relations-between-geometric-objects.html#function_$1',
	'functions-that-create-new-geometries-from-existing-ones.html#function_$1',
	'geometry-property-functions.html#function_$1',
	'gis-wkt-functions.html#function_st-$1',
	'row-subqueries.html',
	'fulltext-search.html#function_match'
];
jush.urls.sqlset = ['https://dev.mysql.com/doc/mysql/en/$key',
	'innodb-parameters.html#sysvar_$1',
	'mysql-cluster-program-options-mysqld.html#option_mysqld_$1', 'mysql-cluster-replication-conflict-resolution.html#option_mysqld_$1', 'mysql-cluster-replication-schema.html', 'mysql-cluster-replication-starting.html', 'mysql-cluster-system-variables.html#sysvar_$1',
	'replication-options-binary-log.html#option_mysqld_$1', 'replication-options-binary-log.html#sysvar_$1', 'replication-options-master.html#sysvar_$1', 'replication-options-slave.html#option_mysqld_log-slave-updates', 'replication-options-slave.html#option_mysqld_$1', 'replication-options-slave.html#sysvar_$1', 'replication-options.html#option_mysqld_$1',
	'server-options.html#option_mysqld_big-tables', 'server-options.html#option_mysqld_$1',
	'server-system-variables.html#sysvar_$1', // previously server-session-variables
	'server-system-variables.html#sysvar_low_priority_updates', 'server-system-variables.html#sysvar_max_join_size', 'server-system-variables.html#sysvar_$1',
	'ssl-options.html#option_general_$1'
];
jush.urls.sqlstatus = ['https://dev.mysql.com/doc/mysql/en/$key',
	'server-status-variables.html#statvar_Com_xxx',
	'server-status-variables.html#statvar_$1'
];

jush.links.sql_sqlset = { 'set-statement.html': /.+/ };

jush.links2.sql = /(\b)(ALTER(?:\s+DEFINER\s*=\s*\S+)?\s+EVENT|(ALTER(?:\s+ONLINE|\s+OFFLINE)?(?:\s+IGNORE)?\s+TABLE)|(ALTER(?:\s+ALGORITHM\s*=\s*(?:UNDEFINED|MERGE|TEMPTABLE))?(?:\s+DEFINER\s*=\s*\S+)?(?:\s+SQL\s+SECURITY\s+(?:DEFINER|INVOKER))?\s+VIEW)|(ANALYZE(?:\s+NO_WRITE_TO_BINLOG|\s+LOCAL)?\s+TABLE)|(CREATE(?:\s+DEFINER\s*=\s*\S+)?\s+EVENT)|(CREATE(?:\s+DEFINER\s*=\s*\S+)?\s+FUNCTION)|(CREATE(?:\s+DEFINER\s*=\s*\S+)?\s+PROCEDURE)|(CREATE(?:\s+ONLINE|\s+OFFLINE)?(?:\s+UNIQUE|\s+FULLTEXT|\s+SPATIAL)?\s+INDEX)|(CREATE(?:\s+TEMPORARY)?\s+TABLE)|(CREATE(?:\s+DEFINER\s*=\s*\S+)?\s+TRIGGER)|(CREATE(?:\s+OR\s+REPLACE)?(?:\s+ALGORITHM\s*=\s*(?:UNDEFINED|MERGE|TEMPTABLE))?(?:\s+DEFINER\s*=\s*\S+)?(?:\s+SQL\s+SECURITY\s+(?:DEFINER|INVOKER))?\s+VIEW)|(DROP(?:\s+ONLINE|\s+OFFLINE)?\s+INDEX)|(DROP(?:\s+TEMPORARY)?\s+TABLE)|(END)|(OPTIMIZE(?:\s+NO_WRITE_TO_BINLOG|\s+LOCAL)?\s+TABLE)|(REPAIR(?:\s+NO_WRITE_TO_BINLOG|\s+LOCAL)?\s+TABLE)|(SET(?:\s+GLOBAL|\s+SESSION)?\s+TRANSACTION\s+ISOLATION\s+LEVEL)|(SHOW(?:\s+FULL)?\s+COLUMNS)|(SHOW(?:\s+STORAGE)?\s+ENGINES)|(SHOW\s+(?:INDEX|INDEXES|KEYS))|(SHOW(?:\s+FULL)?\s+PROCESSLIST)|(SHOW(?:\s+GLOBAL|\s+SESSION)?\s+STATUS)|(SHOW(?:\s+FULL)?\s+TABLES)|(SHOW(?:\s+GLOBAL|\s+SESSION)?\s+VARIABLES)|(ALTER\s+(?:DATABASE|SCHEMA)|ALTER\s+LOGFILE\s+GROUP|ALTER\s+SERVER|ALTER\s+TABLESPACE|BACKUP\s+TABLE|CACHE\s+INDEX|CALL|CHANGE\s+MASTER\s+TO|CHECK\s+TABLE|CHECKSUM\s+TABLE|CREATE\s+(?:DATABASE|SCHEMA)|CREATE\s+LOGFILE\s+GROUP|CREATE\s+SERVER|CREATE\s+TABLESPACE|CREATE\s+USER|DELETE|DESCRIBE|DO|DROP\s+(?:DATABASE|SCHEMA)|DROP\s+EVENT|DROP\s+FUNCTION|DROP\s+PROCEDURE|DROP\s+LOGFILE\s+GROUP|DROP\s+SERVER|DROP\s+TABLESPACE|DROP\s+TRIGGER|DROP\s+USER|DROP\s+VIEW|EXPLAIN|FLUSH|GRANT|HANDLER|HELP|INSERT|INSTALL\s+PLUGIN|JOIN|KILL|LOAD\s+DATA\s+FROM\s+MASTER|LOAD\s+DATA|LOAD\s+INDEX|LOAD\s+XML|PURGE\s+MASTER\s+LOGS|RENAME\s+(?:DATABASE|SCHEMA)|RENAME\s+TABLE|RENAME\s+USER|REPLACE|RESET\s+MASTER|RESET\s+SLAVE|RESIGNAL|RESTORE\s+TABLE|REVOKE|SELECT|SET\s+PASSWORD|SHOW\s+AUTHORS|SHOW\s+BINARY\s+LOGS|SHOW\s+BINLOG\s+EVENTS|SHOW\s+CHARACTER\s+SET|SHOW\s+COLLATION|SHOW\s+CONTRIBUTORS|SHOW\s+CREATE\s+(?:DATABASE|SCHEMA)|SHOW\s+CREATE\s+TABLE|SHOW\s+CREATE\s+VIEW|SHOW\s+(?:DATABASE|SCHEMA)S|SHOW\s+ENGINE|SHOW\s+ERRORS|SHOW\s+GRANTS|SHOW\s+MASTER\s+STATUS|SHOW\s+OPEN\s+TABLES|SHOW\s+PLUGINS|SHOW\s+PRIVILEGES|SHOW\s+SCHEDULER\s+STATUS|SHOW\s+SLAVE\s+HOSTS|SHOW\s+SLAVE\s+STATUS|SHOW\s+TABLE\s+STATUS|SHOW\s+TRIGGERS|SHOW\s+WARNINGS|SHOW|SIGNAL|START\s+SLAVE|STOP\s+SLAVE|UNINSTALL\s+PLUGIN|UNION|UPDATE|USE)|(LOOP|LEAVE|ITERATE|WHILE)|(IF|ELSEIF)|(REPEAT|UNTIL)|(TRUNCATE(?:\s+TABLE)?)|(START\s+TRANSACTION|BEGIN|COMMIT|ROLLBACK)|(SAVEPOINT|ROLLBACK\s+TO\s+SAVEPOINT)|((?:UN)?LOCK\s+TABLES?)|(SET\s+NAMES|SET\s+CHARACTER\s+SET)|(ON\s+DUPLICATE\s+KEY\s+UPDATE)|(IN\s+BOOLEAN\s+MODE|IN\s+NATURAL\s+LANGUAGE\s+MODE|WITH\s+QUERY\s+EXPANSION)|(AUTO_INCREMENT)|(IS|IS\s+NULL)|(BETWEEN|NOT\s+BETWEEN|IN|NOT\s+IN)|(ANY|SOME)|(ALL)|(EXISTS|NOT\s+EXISTS)|(WITH\s+ROLLUP)|(SOUNDS\s+LIKE)|(LIKE|NOT\s+LIKE)|(NOT\s+REGEXP|REGEXP)|(RLIKE)|(NOT|AND|OR|XOR)|(CASE)|(DIV)|(BINARY)|(CURRENT_DATE|CURRENT_TIME|CURRENT_TIMESTAMP|LOCALTIME|LOCALTIMESTAMP|UTC_DATE|UTC_TIME|UTC_TIMESTAMP)|(INTERVAL)|(ACCESSIBLE|ADD|ALTER|ANALYZE|AS|ASC|ASENSITIVE|BEFORE|BOTH|BY|CASCADE|CHANGE|CHARACTER|CHECK|CLOSE|COLLATE|COLUMN|CONDITION|CONSTRAINT|CONTINUE|CONVERT|CREATE|CROSS|CURSOR|DATABASE|DATABASES|DAY_HOUR|DAY_MICROSECOND|DAY_MINUTE|DAY_SECOND|DECLARE|DEFAULT|DELAYED|DESC|DETERMINISTIC|DISTINCT|DISTINCTROW|DROP|DUAL|EACH|ELSE|ENCLOSED|ESCAPED|EXIT|FALSE|FETCH|FLOAT4|FLOAT8|FOR|FORCE|FOREIGN|FROM|FULLTEXT|GROUP|HAVING|HIGH_PRIORITY|HOUR_MICROSECOND|HOUR_MINUTE|HOUR_SECOND|IGNORE|INDEX|INFILE|INNER|INOUT|INSENSITIVE|INT1|INT2|INT3|INT4|INT8|INTO|KEY|KEYS|LEADING|LEFT|LIMIT|LINEAR|LINES|LOAD|LOCK|LONG|LOW_PRIORITY|MASTER_SSL_VERIFY_SERVER_CERT|MATCH|MIDDLEINT|MINUTE_MICROSECOND|MINUTE_SECOND|MODIFIES|NATURAL|NO_WRITE_TO_BINLOG|NULL|OFFSET|ON|OPEN|OPTIMIZE|OPTION|OPTIONALLY|ORDER|OUT|OUTER|OUTFILE|PRECISION|PRIMARY|PROCEDURE|PURGE|RANGE|READ|READS|READ_WRITE|REFERENCES|RELEASE|RENAME|REQUIRE|RESTRICT|RETURN|RIGHT|SCHEMA|SCHEMAS|SECOND_MICROSECOND|SENSITIVE|SEPARATOR|SPATIAL|SPECIFIC|SQL|SQLEXCEPTION|SQLSTATE|SQLWARNING|SQL_BIG_RESULT|SQL_CALC_FOUND_ROWS|SQL_SMALL_RESULT|SSL|STARTING|STRAIGHT_JOIN|TABLE|TERMINATED|THEN|TO|TRAILING|TRIGGER|TRUE|UNDO|UNIQUE|UNLOCK|UNSIGNED|USAGE|USING|VALUES|VARCHARACTER|VARYING|WHEN|WHERE|WITH|WRITE|XOR|YEAR_MONTH|ZEROFILL))\b(?!\()|\b(bit|tinyint|bool|boolean|smallint|mediumint|int|integer|bigint|float|double\s+precision|double|real|decimal|dec|numeric|fixed|(date|datetime|timestamp|time|year)|(char|varchar|binary|varbinary|tinyblob|tinytext|blob|text|mediumblob|mediumtext|longblob|longtext|enum|set)|(geometry|point|linestring|polygon|multipoint|multilinestring|multipolygon|geometrycollection)|(mod)|(CURRENT_USER)|(InnoDB|MyISAM|MEMORY|CSV|ARCHIVE|BLACKHOLE|MERGE|FEDERATED)|(MRG_MyISAM)|(PARTITION\s+BY\s+RANGE)|(PARTITION\s+BY\s+LIST)|(PARTITION\s+BY\s+COLUMNS)|(PARTITION\s+BY\s+HASH)|(PARTITION\s+BY\s+LINEAR\s+HASH)|(PARTITION\s+BY(?:\s+LINEAR)?\s+KEY))\b|\b(coalesce|greatest|isnull|interval|least|(if|ifnull|nullif)|(ascii|bin|bit_length|char|char_length|character_length|concat|concat_ws|conv|elt|export_set|field|find_in_set|format|hex|insert|instr|lcase|left|length|load_file|locate|lower|lpad|ltrim|make_set|mid|oct|octet_length|ord|position|quote|repeat|replace|reverse|right|rpad|rtrim|soundex|sounds_like|space|substr|substring|substring_index|trim|ucase|unhex|upper)|(strcmp)|(abs|acos|asin|atan|atan2|ceil|ceiling|cos|cot|crc32|degrees|exp|floor|ln|log|log2|log10|pi|pow|power|radians|rand|round|sign|sin|sqrt|tan|truncate)|(adddate|addtime|convert_tz|curdate|curtime|date|datediff|date_add|date_format|date_sub|day|dayname|dayofmonth|dayofweek|dayofyear|extract|from_days|from_unixtime|get_format|hour|last_day|makedate|maketime|microsecond|minute|month|monthname|now|period_add|period_diff|quarter|second|sec_to_time|str_to_date|subdate|subtime|sysdate|time|timediff|timestamp|timestampadd|timestampdiff|time_format|time_to_sec|to_days|to_seconds|unix_timestamp|week|weekday|weekofyear|year|yearweek)|(cast|convert)|(extractvalue|updatexml)|(bit_count)|(aes_encrypt|aes_decrypt|compress|decode|encode|des_decrypt|des_encrypt|encrypt|md5|old_password|password|sha|sha1|uncompress|uncompressed_length)|(benchmark|charset|coercibility|collation|connection_id|database|found_rows|last_insert_id|row_count|schema|session_user|system_user|user|version)|(default|get_lock|inet_aton|inet_ntoa|is_free_lock|is_used_lock|master_pos_wait|name_const|release_lock|sleep|uuid|uuid_short|values)|(avg|bit_and|bit_or|bit_xor|count|count_distinct|group_concat|min|max|std|stddev|stddev_pop|stddev_samp|sum|var_pop|var_samp|variance)|(asbinary|aswkb)|(astext|aswkt)|(mbrcontains|mbrdisjoint|mbrequal|mbrintersects|mbroverlaps|mbrtouches|mbrwithin|contains|crosses|disjoint|equals|intersects|overlaps|touches|within)|(buffer|convexhull|difference|intersection|symdifference)|(dimension|envelope|geometrytype|srid|boundary|isempty|issimple|x|y|endpoint|glength|numpoints|pointn|startpoint|isring|isclosed|area|exteriorring|interiorringn|numinteriorrings|centroid|geometryn|numgeometries)|(geomcollfromtext|geomfromtext|linefromtext|mlinefromtext|mpointfromtext|mpolyfromtext|pointfromtext|polyfromtext|bdmpolyfromtext|bdpolyfromtext|geomcollfromwkb|geomfromwkb|linefromwkb|mlinefromwkb|mpointfromwkb|mpolyfromwkb|pointfromwkb|polyfromwkb|bdmpolyfromwkb|bdpolyfromwkb|geometrycollection|linestring|multilinestring|multipoint|multipolygon|point|polygon)|(row)|(match|against))(\s*\(|$)/gi; // collisions: char, set, union(), allow parenthesis - IN, ANY, ALL, SOME, NOT, AND, OR, XOR
jush.links2.sqlset = /(\b)(ignore_builtin_innodb|innodb_adaptive_hash_index|innodb_additional_mem_pool_size|innodb_autoextend_increment|innodb_autoinc_lock_mode|innodb_buffer_pool_awe_mem_mb|innodb_buffer_pool_size|innodb_commit_concurrency|innodb_concurrency_tickets|innodb_data_file_path|innodb_data_home_dir|innodb_doublewrite|innodb_fast_shutdown|innodb_file_io_threads|innodb_file_per_table|innodb_flush_log_at_trx_commit|innodb_flush_method|innodb_force_recovery|innodb_checksums|innodb_lock_wait_timeout|innodb_locks_unsafe_for_binlog|innodb_log_arch_dir|innodb_log_archive|innodb_log_buffer_size|innodb_log_file_size|innodb_log_files_in_group|innodb_log_group_home_dir|innodb_max_dirty_pages_pct|innodb_max_purge_lag|innodb_mirrored_log_groups|innodb_open_files|innodb_rollback_on_timeout|innodb_stats_on_metadata|innodb_support_xa|innodb_sync_spin_loops|innodb_table_locks|innodb_thread_concurrency|innodb_thread_sleep_delay|innodb_use_legacy_cardinality_algorithm|(ndb[-_]batch[-_]size)|(ndb[-_]log[-_]update[-_]as[-_]write|ndb_log_updated_only)|(ndb_log_orig)|(slave[-_]allow[-_]batching)|(have_ndbcluster|multi_range_count|ndb_autoincrement_prefetch_sz|ndb_cache_check_time|ndb_extra_logging|ndb_force_send|ndb_use_copying_alter_table|ndb_use_exact_count|ndb_wait_connected)|(log[-_]bin[-_]trust[-_]function[-_]creators|log[-_]bin)|(binlog_cache_size|max_binlog_cache_size|max_binlog_size|sync_binlog)|(auto_increment_increment|auto_increment_offset)|(ndb_log_empty_epochs)|(log[-_]slave[-_]updates|report[-_]host|report[-_]password|report[-_]port|report[-_]user|slave[-_]net[-_]timeout|slave[-_]skip[-_]errors)|(init_slave|rpl_recovery_rank|slave_compressed_protocol|slave_exec_mode|slave_transaction_retries|sql_slave_skip_counter)|(master[-_]bind|slave[-_]load[-_]tmpdir|server[-_]id)|(sql_big_tables)|(basedir|big[-_]tables|binlog[-_]format|collation[-_]server|datadir|debug|delay[-_]key[-_]write|engine[-_]condition[-_]pushdown|event[-_]scheduler|general[-_]log|character[-_]set[-_]filesystem|character[-_]set[-_]server|character[-_]sets[-_]dir|init[-_]file|language|large[-_]pages|log[-_]error|log[-_]output|log[-_]queries[-_]not[-_]using[-_]indexes|log[-_]slow[-_]queries|log[-_]warnings|log|low[-_]priority[-_]updates|memlock|min[-_]examined[-_]row[-_]limit|old[-_]passwords|open[-_]files[-_]limit|pid[-_]file|port|safe[-_]show[-_]database|secure[-_]auth|secure[-_]file[-_]priv|skip[-_]external[-_]locking|skip[-_]networking|skip[-_]show[-_]database|slow[-_]query[-_]log|socket|sql[-_]mode|tmpdir|version)|(autocommit|error_count|foreign_key_checks|identity|insert_id|last_insert_id|profiling|profiling_history_size|rand_seed1|rand_seed2|sql_auto_is_null|sql_big_selects|sql_buffer_result|sql_log_bin|sql_log_off|sql_log_update|sql_notes|sql_quote_show_create|sql_safe_updates|sql_warnings|timestamp|unique_checks|warning_count)|(sql_low_priority_updates)|(sql_max_join_size)|(automatic_sp_privileges|back_log|bulk_insert_buffer_size|collation_connection|collation_database|completion_type|concurrent_insert|connect_timeout|date_format|datetime_format|default_week_format|delayed_insert_limit|delayed_insert_timeout|delayed_queue_size|div_precision_increment|expire_logs_days|flush|flush_time|ft_boolean_syntax|ft_max_word_len|ft_min_word_len|ft_query_expansion_limit|ft_stopword_file|general_log_file|group_concat_max_len|have_archive|have_blackhole_engine|have_compress|have_crypt|have_csv|have_dynamic_loading|have_example_engine|have_federated_engine|have_geometry|have_innodb|have_isam|have_merge_engine|have_openssl|have_partitioning|have_query_cache|have_raid|have_row_based_replication|have_rtree_keys|have_ssl|have_symlink|hostname|character_set_client|character_set_connection|character_set_database|character_set_results|character_set_system|init_connect|interactive_timeout|join_buffer_size|keep_files_on_create|key_buffer_size|key_cache_age_threshold|key_cache_block_size|key_cache_division_limit|large_page_size|lc_time_names|license|local_infile|locked_in_memory|log_bin|long_query_time|lower_case_file_system|lower_case_table_names|max_allowed_packet|max_connect_errors|max_connections|max_delayed_threads|max_error_count|max_heap_table_size|max_insert_delayed_threads|max_join_size|max_length_for_sort_data|max_prepared_stmt_count|max_relay_log_size|max_seeks_for_key|max_sort_length|max_sp_recursion_depth|max_tmp_tables|max_user_connections|max_write_lock_count|myisam_data_pointer_size|myisam_max_sort_file_size|myisam_recover_options|myisam_repair_threads|myisam_sort_buffer_size|myisam_stats_method|myisam_use_mmap|named_pipe|net_buffer_length|net_read_timeout|net_retry_count|net_write_timeout|new|old|optimizer_prune_level|optimizer_search_depth|optimizer_switch|plugin_dir|preload_buffer_size|prepared_stmt_count|protocol_version|pseudo_thread_id|query_alloc_block_size|query_cache_limit|query_cache_min_res_unit|query_cache_size|query_cache_type|query_cache_wlock_invalidate|query_prealloc_size|range_alloc_block_size|read_buffer_size|read_only|read_rnd_buffer_size|relay_log_purge|relay_log_space_limit|shared_memory|shared_memory_base_name|slow_launch_time|slow_query_log_file|sort_buffer_size|sql_select_limit|storage_engine|sync_frm|system_time_zone|table_cache|table_definition_cache|table_lock_wait_timeout|table_open_cache|table_type|thread_cache_size|thread_concurrency|thread_handling|thread_stack|time_format|time_zone|timed_mutexes|tmp_table_size|transaction_alloc_block_size|transaction_prealloc_size|tx_isolation|updatable_views_with_limit|version_comment|version_compile_machine|version_compile_os|wait_timeout)|(ssl[-_]ca|ssl[-_]capath|ssl[-_]cert|ssl[-_]cipher|ssl[-_]key))((?!-)\b)/gi;
jush.links2.sqlstatus = /()(Com_.+|(.+))()/gi;
jush.tr.sqlite = { sqlite_apo: /'/, sqlite_quo: /"/, bra: /\[/, bac: /`/, one: /--/, com: /\/\*/, sql_var: /[:@$]/, sqlite_sqliteset: /(\b)(PRAGMA)(\s+)/i, num: jush.num };
jush.tr.sqlite_sqliteset = { sqlite_apo: /'/, sqlite_quo: /"/, bra: /\[/, bac: /`/, one: /--/, com: /\/\*/, num: jush.num, _1: /;|$/ };
jush.tr.sqliteset = { _0: /$/ };
jush.tr.sqlitestatus = { _0: /$/ };

jush.urls.sqlite_sqliteset = 'https://www.sqlite.org/$key';
jush.urls.sqlite = ['https://www.sqlite.org/$key',
	'lang_$1.html', 'lang_createvtab.html', 'lang_transaction.html',
	'lang_createindex.html', 'lang_createtable.html', 'lang_createtrigger.html', 'lang_createview.html',
	'',
	'lang_expr.html#$1', 'lang_corefunc.html#$1', 'lang_datefunc.html#$1', 'lang_aggfunc.html#$1'
];
jush.urls.sqliteset = ['https://www.sqlite.org/pragma.html#$key',
	'pragma_$1'
];
jush.urls.sqlitestatus = ['https://www.sqlite.org/compile.html#$key',
	'$1'
];

jush.links.sqlite_sqliteset = { 'pragma.html': /.+/ };

jush.links2.sqlite = /(\b)(ALTER\s+TABLE|ANALYZE|ATTACH|COPY|DELETE|DETACH|DROP\s+INDEX|DROP\s+TABLE|DROP\s+TRIGGER|DROP\s+VIEW|EXPLAIN|INSERT|CONFLICT|REINDEX|REPLACE|SELECT|UPDATE|TRANSACTION|VACUUM|(CREATE\s+VIRTUAL\s+TABLE)|(BEGIN|COMMIT|ROLLBACK)|(CREATE(?:\s+UNIQUE)?\s+INDEX)|(CREATE(?:\s+TEMP|\s+TEMPORARY)?\s+TABLE)|(CREATE(?:\s+TEMP|\s+TEMPORARY)?\s+TRIGGER)|(CREATE(?:\s+TEMP|\s+TEMPORARY)?\s+VIEW)|(ABORT|ACTION|ADD|AFTER|ALL|AS|ASC|AUTOINCREMENT|BEFORE|BY|CASCADE|CHECK|COLUMN|CONSTRAINT|CROSS|CURRENT_DATE|CURRENT_TIME|CURRENT_TIMESTAMP|DATABASE|DEFAULT|DEFERRABLE|DEFERRED|DESC|DISTINCT|EACH|END|EXCEPT|EXCLUSIVE|FAIL|FOR|FOREIGN|FROM|FULL|GROUP|HAVING|IF|IGNORE|IMMEDIATE|INDEXED|INITIALLY|INNER|INSTEAD|INTERSECT|INTO|IS|JOIN|KEY|LEFT|LIMIT|NATURAL|NO|NOT|NOTNULL|NULL|OF|OFFSET|ON|ORDER|OUTER|PLAN|PRAGMA|PRIMARY|QUERY|RAISE|REFERENCES|RELEASE|RENAME|RESTRICT|RIGHT|ROW|SAVEPOINT|SET|TEMPORARY|TO|UNION|UNIQUE|USING|VALUES|WHERE)|(like|glob|regexp|match|escape|isnull|isnotnull|between|exists|case|when|then|else|cast|collate|in|and|or|not))\b|\b(abs|coalesce|glob|ifnull|hex|last_insert_rowid|length|like|load_extension|lower|nullif|quote|random|randomblob|round|soundex|sqlite_version|substr|typeof|upper|(date|time|datetime|julianday|strftime)|(avg|count|max|min|sum|total))(\s*\(|$)/gi; // collisions: min, max, end, like, glob
jush.links2.sqliteset = /(\b)(auto_vacuum|cache_size|case_sensitive_like|count_changes|default_cache_size|empty_result_callbacks|encoding|foreign_keys|full_column_names|fullfsync|incremental_vacuum|journal_mode|journal_size_limit|legacy_file_format|locking_mode|page_size|max_page_count|read_uncommitted|recursive_triggers|reverse_unordered_selects|secure_delete|short_column_names|synchronous|temp_store|temp_store_directory|collation_list|database_list|foreign_key_list|freelist_count|index_info|index_list|page_count|table_info|schema_version|compile_options|integrity_check|quick_check|parser_trace|vdbe_trace|vdbe_listing)(\b)/gi;
jush.links2.sqlitestatus = /()(.+)()/g;jush.tr.mssql = { sqlite_apo: /'/, sqlite_quo: /"/, one: /--/, com: /\/\*/, mssql_bra: /\[/, num: jush.num }; // QUOTED IDENTIFIER = OFF
jush.tr.mssql_bra = { _0: /]]/, _1: /]/ };

jush.urls.mssql = ['https://msdn.microsoft.com/library/$key.aspx',
	'ms181700', 'ms178543', 'ms188372', 'ms189526', 'ms186865', 'ms178578', 'ms174387', 'ms190337', 'ms190487', 'ms187804', 'ms187377', 'ms188386', 'ms188929', 'ms187922', 'ms188362', 'ms177603', 'ms181271', 'ms188365', 'ms181765', 'ms187368', 'ms176089', 'ms188748', 'ms175035', 'ms188387', 'ms177938', 'ms184391', 'ms178628', 'ms190295', 'ms181708', 'ms174366', 'bb630352', 'ms187819', 'bb677335', 'bb630289', 'ms188782', 'ms187746', 'ms188927', 'ms180169', 'ms189835', 'ms188338', 'ms189748', 'ms182587', 'ms182706', 'ms176030', 'ms177521', 'ms188055', 'ms188332', 'ms181362', 'ms188336', 'ms180152', 'ms173773', 'ms173812', 'ms177634', 'cc280766', 'cc280487', 'ms178624', 'ms188037', 'ms180188', 'ms187965', 'ms177673', 'ms180199', 'bb677290', 'ms186775', 'ms182717', 'ms177682', 'ms174335', 'ms187745', 'ms188029', 'ms188795', 'ms173730', 'ms186764', 'ms180016', 'ms179859', 'bb510625', 'ms179882', 'ms174987', 'ms186939', 'ms189455', 'ms187993', 'ms190500', 'ms174433', 'ms190499', 'ms190322', 'ms188361', 'ms188385', 'ms177564', 'ms189461', 'ms176047', 'ms190372', 'ms186336', 'ms187972', 'ms174998', 'ms178632', 'ms187728', 'ms181299', 'ms174973', 'ms182776', 'ms188378', 'ms189499', 'ms188407', 'ms190356', 'ms188767', 'ms182418', 'ms175064', 'ms173829', 'bb677243', 'ms189463', 'ms175976', 'ms177570', 'ms180026', 'ms187942', 'ms177523', 'ms187348', 'ms189466', 'ms188366', 'ms186290', 'ms187331', 'ms188047', 'ms178642', 'ms175972', 'ms177607', 'ms186838', 'ms189797',
	'ms182741', 'ms181491', 'ms189524', 'ms174430', 'bb934170', 'ms187798', 'ms178528', 'ms189522', 'bb677184', 'ms176061', 'cc280404', 'bb677241', 'ms173565', 'ms181591', 'ms189453', 'bb677289', 'ms189520', 'ms187317', 'cc280405', 'ms186755', 'ms188783', 'ms189751', 'ms174382', 'ms187744', 'ms187802', 'ms179854', 'ms187926', 'ms190495', 'ms178024', 'bb895329', 'ms187936', 'ms186742', 'ms188064', 'ms189462', 'cc280448', 'cc280767', 'ms190332', 'ms188038', 'ms188357', 'ms177544', 'ms174979', 'ms189799', 'ms175007', 'ms173463', 'ms187956', 'bb934146', 'ms176009',
	'ms186847', 'ms177517', 'ms177514', 'ms188389', 'bb964728', 'ms179906', 'ms190475', 'ms189450', 'bb677309', 'ms178613', 'cc280479', 'bb630256', 'ms188747', 'ms181586', 'ms174414', 'bb630257', 'ms188403', 'ms184393', 'cc280482', 'ms190290', 'ms176118', 'ms188012', 'ms180071', 'ms186728', 'ms187759', 'ms181249', 'ms174969', 'ms190480', 'ms177539', 'bb933779', 'ms174988', 'ms189449', 'ms186791', 'ms186751', 'cc280899', 'cc280603', 'ms174990', 'ms186977', 'ms175075', 'ms182698', 'ms174996', 'ms173790', 'ms173497', 'ms174407', 'ms189438', 'ms173492', 'bb933867', 'ms189448',
	'ms188900', 'ms186711', 'ms187311', 'ms187359', 'bb933778', 'ms189511', 'ms187923', 'bb677336', 'ms174269', 'cc280645', 'bb630389', 'ms186332', 'bb630368', 'ms176095', 'ms188359', 'cc280871', 'ms186967', 'ms188388', 'ms189828', 'ms186937', 'ms187345', 'ms186307', 'ms190347', 'ms189762', 'ms189529', 'ms190363', 'bb934013', 'bb934024', 'ms189775', 'ms187353', 'ms173423', 'cc280563', 'cc280682', 'ms176036', 'ms187788', 'ms189440', 'ms190273', 'ms176072', 'ms176060', 'ms173846', 'bb895361', 'ms189778',
	'ms189800', 'ms178627', 'ms189749', 'ms178647', 'ms189770', 'ms177545', 'ms181581', 'ms188750', 'ms189508', 'cc627393', 'ms181746', 'ms173854', 'ms177677', 'ms173784', 'ms187928', 'ms189818', 'ms187324', 'ms180142', 'ms187323', 'ms186323', 'ms189788', 'ms188920', 'ms190349', 'ms190305', 'ms188732', 'ms174974', 'ms174968', 'ms186329', 'bb895240', 'ms187787', 'ms189760', 'ms180125', 'ms189534', 'ms188919', 'ms188921', 'ms175997', 'ms190317', 'cc627408', 'ms187352', 'ms188751', 'ms176050', 'ms177609', 'ms187319', 'ms176049', 'ms186823', 'ms173486', 'ms186819', 'ms189794', 'ms174395', 'ms174420', 'ms176052', 'ms186274', 'ms189753', 'ms188796', 'ms189507', 'ms178601', 'ms181860', 'ms365420', 'ms182559', 'ms188910', 'ms178566', 'ms173825', 'ms188753', 'ms186950', 'ms188061', 'ms174361', 'ms190357', 'ms178600', 'ms190358', 'ms175069', 'ms188398', 'ms178567', 'ms180031', 'ms173781', 'ms179857', 'ms182063', 'ms186275', 'ms181399', 'ms186980', 'ms176088', 'ms188069', 'ms188401', 'ms178531', 'ms186788', 'ms176078', 'ms177652', 'ms190370', 'ms188418', 'bb934014', 'ms181825', 'ms174960', 'ms188383', 'ms178635', 'ms178544', 'bb510624', 'ms187718', 'ms189802', 'ms174415', 'ms177605', 'ms178598', 'ms175098', 'ms189795', 'ms189834', 'ms186773', 'ms187729', 'ms178545', 'ms186271', 'cc627401', 'ms176015', 'ms187347', 'ms184325', 'ms186272', 'ms187385', 'ms189457', 'cc645960', 'ms177601', 'ms190329', 'ms190319', 'ms175121', 'ms345412', 'ms174400', 'ms177827', 'ms187751', 'ms179916', 'bb839514', 'ms187813', 'ms182673', 'ms190348', 'ms189786', 'ms175126', 'ms177562', 'ms176090', 'ms190328', 'ms186301', 'bb326599', 'ms176105', 'ms188390', 'ms179856', 'ms188427', 'ms190312', 'ms186918', 'bb895328', 'ms189492', 'ms188006', 'bb895239', 'ms188395', 'ms186915', 'ms189512', 'ms174276', 'ms189830', 'dd822792', 'dd822791', 'ms176114', 'ms189742', 'ms178592', 'ms177610', 'ms176102', 'ms187365', 'ms186963', 'ms176069', 'ms186862', 'ms174383', 'ms180040', 'ms177532', 'ms175003', 'ms186734', 'ms181406', 'ms178660', 'ms188797', 'ms175068', 'ms190315', 'ms174396', 'ms177587', 'ms175001', 'ms186297', 'ms188420', 'ms187913', 'ms187920', 'ms188377', 'ms187384', 'ms187950', 'ms178550', 'ms176108', 'ms173569', 'ms190330', 'ms190474', 'ms176080', 'ms189527', 'ms188043', 'ms187748', 'ms187810', 'ms176042', 'ms187934', 'ms179889', 'ms174427', 'bb677244', 'bb630353', 'bb677334', 'cc645882', 'bb630387', 'ms179930', 'ms190338', 'ms186881', 'ms176068', 'ms187362', 'bb630335', 'ms182737', 'ms181628', 'ms189750', 'ms188419', 'ms180059', 'ms187326', 'ms180055', 'ms186738', 'ms181466', 'ms188014', 'ms188735', 'ms178631', 'ms187791', 'ms187339', 'ms190316', 'ms186313'
];

jush.links2.mssql = /(\b)(ADD(?:\s+COUNTER)?\s+SIGNATURE|(ALL)|(AND)|(ANY)|(BACKUP)|(BACKUP\s+CERTIFICATE)|(BACKUP\s+MASTER\s+KEY)|(BACKUP\s+SERVICE\s+MASTER\s+KEY)|(BEGIN)|(BEGIN\s+CONVERSATION\s+TIMER)|(BEGIN\s+DIALOG)|(BEGIN\s+DISTRIBUTED\s+(?:TRANSACTION|TRAN))|(BEGIN\s+(?:TRANSACTION|TRAN))|(BETWEEN)|((?:var)?binary)|(bit)|(BREAK)|(BULK\s+INSERT)|(CASE)|(CATCH)|((?:var)?char)|(CHECKPOINT)|(CLOSE)|(CLOSE\s+MASTER\s+KEY)|(CLOSE\s+(?:SYMMETRIC\s+KEY|ALL\s+SYMMETRIC\s+KEYS))|(COLLATE)|(COMMIT)|(COMMIT\s+(?:TRANSACTION|TRAN))|(COMPUTE)|(CONTINUE)|(date)|(datetime)|(datetime2)|(datetimeoffset)|(DEALLOCATE)|(decimal|numeric)|(DECLARE)|(DECLARE\s+CURSOR)|(DELETE)|(DENY)|(DISABLE\s+TRIGGER)|(ELSE)|(ENABLE\s+TRIGGER)|(END)|(END\s+CONVERSATION)|(EXCEPT|INTERSECT)|(EXECUTE|EXEC)|((?:EXECUTE|EXEC)\s+AS)|(EXISTS)|(FETCH)|(float|real)|(FOR)|(FROM)|(geography)|(geometry)|(GET\s+CONVERSATION\s+GROUP)|(GO)|(GOTO)|(GRANT)|(GROUP\s+BY)|(HAVING)|(hierarchyid)|(IDENTITY)|(IF)|(IN)|(INSERT)|((?:big|small|tiny)?int)|(INTO)|(IS(?:\s+NOT)?\s+NULL)|(KILL)|(KILL\s+QUERY\s+NOTIFICATION\s+SUBSCRIPTION)|(KILL\s+STATS\s+JOB)|(LIKE)|(MERGE)|((?:small)?money)|(MOVE\s+CONVERSATION)|(nchar|nvarchar)|(NOT)|(ntext|text|image)|(OPEN)|(OPEN\s+MASTER\s+KEY)|(OPEN\s+SYMMETRIC\s+KEY)|(OPTION)|(OR)|(ORDER\s+BY)|(OUTPUT)|(OVER)|(PRINT)|(RESTORE)|(RESTORE\s+MASTER\s+KEY)|(RESTORE\s+SERVICE\s+MASTER\s+KEY)|(RETURN)|(REVERT)|(REVOKE)|(ROLLBACK\s+TRANSACTION)|(ROLLBACK\s+WORK)|(rowversion)|(SAVE\s+TRANSACTION)|(SELECT)|(SEND)|(SET)|(SHUTDOWN)|(smalldatetime)|(SOME|ANY)|(sql_variant)|(time)|(TOP)|(TRY)|(TRUNCATE\s+TABLE)|(UNION)|(uniqueidentifier)|(UPDATE)|(UPDATE\s+STATISTICS)|(UPDATETEXT)|(USE)|(VAR)|(WAITFOR)|(WHERE)|(WHILE)|(WITH)|(WITH\s+XMLNAMESPACES)|(WRITETEXT)|(XACT_STATE)|(CREATE\s+AGGREGATE)|(CREATE\s+APPLICATION\s+ROLE)|(CREATE\s+ASSEMBLY)|(CREATE\s+ASYMMETRIC\s+KEY)|(CREATE\s+BROKER\s+PRIORITY)|(CREATE\s+CERTIFICATE)|(CREATE\s+CONTRACT)|(CREATE\s+CREDENTIAL)|(CREATE\s+CRYPTOGRAPHIC\s+PROVIDER)|(CREATE\s+DATABASE)|(CREATE\s+DATABASE\s+AUDIT\s+SPECIFICATION)|(CREATE\s+DATABASE\s+ENCRYPTION\s+KEY)|(CREATE\s+DEFAULT)|(CREATE\s+ENDPOINT)|(CREATE\s+EVENT\s+NOTIFICATION)|(CREATE\s+EVENT\s+SESSION)|(CREATE\s+FULLTEXT\s+CATALOG)|(CREATE\s+FULLTEXT\s+INDEX)|(CREATE\s+FULLTEXT\s+STOPLIST)|(CREATE\s+FUNCTION)|(CREATE(?:\s+UNIQUE)?\s+INDEX)|(CREATE\s+LOGIN)|(CREATE\s+MASTER\s+KEY)|(CREATE\s+MESSAGE\s+TYPE)|(CREATE\s+PARTITION\s+FUNCTION)|(CREATE\s+PARTITION\s+SCHEME)|(CREATE\s+PROCEDURE)|(CREATE\s+QUEUE)|(CREATE\s+REMOTE\s+SERVICE\s+BINDING)|(CREATE\s+RESOURCE\s+POOL)|(CREATE\s+ROLE)|(CREATE\s+ROUTE)|(CREATE\s+RULE)|(CREATE\s+SCHEMA)|(CREATE\s+SERVER\s+AUDIT)|(CREATE\s+SERVER\s+AUDIT\s+SPECIFICATION)|(CREATE\s+SERVICE)|(CREATE\s+STATISTICS)|(CREATE\s+SYMMETRIC\s+KEY)|(CREATE\s+SYNONYM)|(CREATE\s+TABLE)|(CREATE\s+TRIGGER)|(CREATE\s+TYPE)|(CREATE\s+USER)|(CREATE\s+VIEW)|(CREATE\s+WORKLOAD\s+GROUP)|(CREATE\s+XML\s+SCHEMA\s+COLLECTION)|(DROP\s+AGGREGATE)|(DROP\s+APPLICATION\s+ROLE)|(DROP\s+ASSEMBLY)|(DROP\s+ASYMMETRIC\s+KEY)|(DROP\s+BROKER\s+PRIORITY)|(DROP\s+CERTIFICATE)|(DROP\s+CONTRACT)|(DROP\s+CREDENTIAL)|(DROP\s+CRYPTOGRAPHIC\s+PROVIDER)|(DROP\s+DATABASE)|(DROP\s+DATABASE\s+AUDIT\s+SPECIFICATION)|(DROP\s+DATABASE\s+ENCRYPTION\s+KEY)|(DROP\s+DEFAULT)|(DROP\s+ENDPOINT)|(DROP\s+EVENT\s+NOTIFICATION)|(DROP\s+EVENT\s+SESSION)|(DROP\s+FULLTEXT\s+CATALOG)|(DROP\s+FULLTEXT\s+INDEX)|(DROP\s+FULLTEXT\s+STOPLIST)|(DROP\s+FUNCTION)|(DROP\s+INDEX)|(DROP\s+LOGIN)|(DROP\s+MASTER\s+KEY)|(DROP\s+MESSAGE\s+TYPE)|(DROP\s+PARTITION\s+FUNCTION)|(DROP\s+PARTITION\s+SCHEME)|(DROP\s+PROCEDURE)|(DROP\s+QUEUE)|(DROP\s+REMOTE\s+SERVICE\s+BINDING)|(DROP\s+RESOURCE\s+POOL)|(DROP\s+ROLE)|(DROP\s+ROUTE)|(DROP\s+RULE)|(DROP\s+SCHEMA)|(DROP\s+SERVER\s+AUDIT)|(DROP\s+SERVER\s+AUDIT\s+SPECIFICATION)|(DROP\s+SERVICE)|(DROP\s+SIGNATURE)|(DROP\s+STATISTICS)|(DROP\s+SYMMETRIC\s+KEY)|(DROP\s+SYNONYM)|(DROP\s+TABLE)|(DROP\s+TRIGGER)|(DROP\s+TYPE)|(DROP\s+USER)|(DROP\s+VIEW)|(DROP\s+WORKLOAD\s+GROUP)|(DROP\s+XML\s+SCHEMA\s+COLLECTION)|(ALTER\s+APPLICATION\s+ROLE)|(ALTER\s+ASSEMBLY)|(ALTER\s+ASYMMETRIC\s+KEY)|(ALTER\s+AUTHORIZATION)|(ALTER\s+BROKER\s+PRIORITY)|(ALTER\s+CERTIFICATE)|(ALTER\s+CREDENTIAL)|(ALTER\s+CRYPTOGRAPHIC\s+PROVIDER)|(ALTER\s+DATABASE)|(ALTER\s+DATABASE\s+AUDIT\s+SPECIFICATION)|(ALTER\s+DATABASE\s+ENCRYPTION\s+KEY)|(ALTER\s+ENDPOINT)|(ALTER\s+EVENT\s+SESSION)|(ALTER\s+FULLTEXT\s+CATALOG)|(ALTER\s+FULLTEXT\s+INDEX)|(ALTER\s+FULLTEXT\s+STOPLIST)|(ALTER\s+FUNCTION)|(ALTER\s+INDEX)|(ALTER\s+LOGIN)|(ALTER\s+MASTER\s+KEY)|(ALTER\s+MESSAGE\s+TYPE)|(ALTER\s+PARTITION\s+FUNCTION)|(ALTER\s+PARTITION\s+SCHEME)|(ALTER\s+PROCEDURE)|(ALTER\s+QUEUE)|(ALTER\s+REMOTE\s+SERVICE\s+BINDING)|(ALTER\s+RESOURCE\s+GOVERNOR)|(ALTER\s+RESOURCE\s+POOL)|(ALTER\s+ROLE)|(ALTER\s+ROUTE)|(ALTER\s+SCHEMA)|(ALTER\s+SERVER\s+AUDIT)|(ALTER\s+SERVER\s+AUDIT\s+SPECIFICATION)|(ALTER\s+SERVICE)|(ALTER\s+SERVICE\s+MASTER\s+KEY)|(ALTER\s+SYMMETRIC\s+KEY)|(ALTER\s+TABLE)|(ALTER\s+TRIGGER)|(ALTER\s+USER)|(ALTER\s+VIEW)|(ALTER\s+WORKLOAD\s+GROUP)|(ALTER\s+XML\s+SCHEMA\s+COLLECTION))\b|\b(ABS|(ACOS)|(APPLOCK_MODE)|(APPLOCK_TEST)|(APP_NAME)|(ASCII)|(ASIN)|(ASSEMBLYPROPERTY)|(ASYMKEY_ID)|(ASYMKEYPROPERTY)|(ATAN)|(ATN2)|(AVG)|(BINARY_CHECKSUM )|(CAST|CONVERT)|(CEILING)|(CertProperty)|(Cert_ID)|(CHAR)|(CHARINDEX)|(CHECKSUM)|(CHECKSUM_AGG)|(COALESCE)|(COLLATIONPROPERTY)|(COL_LENGTH)|(COL_NAME)|(COLUMNPROPERTY)|(COLUMNS_UPDATED)|(CONNECTIONPROPERTY)|(CONTAINS)|(CONTAINSTABLE)|(CONTEXT_INFO)|(CONVERT)|(COS)|(COT)|(COUNT)|(COUNT_BIG)|(CRYPT_GEN_RANDOM)|(CURRENT_REQUEST_ID)|(CURRENT_TIMESTAMP)|(CURRENT_USER)|(CURSOR_STATUS)|(DATABASE_PRINCIPAL_ID)|(DATABASEPROPERTY)|(DATABASEPROPERTYEX)|(DATALENGTH)|(DATEADD)|(DATEDIFF)|(DATENAME)|(DATEPART)|(DAY)|(DB_ID)|(DB_NAME)|(DBCC)|(DECRYPTBYASYMKEY)|(DECRYPTBYCERT)|(DECRYPTBYKEY)|(DECRYPTBYKEYAUTOASYMKEY)|(DECRYPTBYKEYAUTOCERT)|(DECRYPTBYPASSPHRASE)|(DEGREES)|(DENSE_RANK)|(DIFFERENCE)|(ENCRYPTBYASYMKEY)|(ENCRYPTBYCERT)|(ENCRYPTBYKEY)|(ENCRYPTBYPASSPHRASE)|(ERROR_LINE)|(ERROR_MESSAGE)|(ERROR_NUMBER)|(ERROR_PROCEDURE)|(ERROR_SEVERITY)|(ERROR_STATE)|(EVENTDATA)|(EXP)|(FILE_ID)|(FILE_IDEX)|(FILE_NAME)|(FILEGROUP_ID)|(FILEGROUP_NAME)|(FILEGROUPPROPERTY)|(FILEPROPERTY)|(FLOOR)|(FORMATMESSAGE)|(FREETEXT)|(FREETEXTTABLE)|(FULLTEXTCATALOGPROPERTY)|(FULLTEXTSERVICEPROPERTY)|(GET_FILESTREAM_TRANSACTION_CONTEXT)|(GET_TRANSMISSION_STATUS)|(GETANSINULL)|(GETDATE)|(GETUTCDATE)|(GROUPING)|(GROUPING_ID)|(HAS_DBACCESS)|(HAS_PERMS_BY_NAME)|(HASHBYTES)|(HOST_ID)|(HOST_NAME)|(IDENT_CURRENT)|(IDENT_INCR)|(IDENT_SEED)|(INDEXKEY_PROPERTY)|(INDEXPROPERTY)|(INDEX_COL)|(IS_MEMBER)|(IS_OBJECTSIGNED)|(IS_SRVROLEMEMBER)|(ISDATE)|(ISNULL)|(ISNUMERIC)|(Key_GUID)|(Key_ID)|(KEY_NAME)|(LEFT)|(LEN)|(LOG)|(LOG10)|(LOGINPROPERTY)|(LOWER)|(LTRIM)|(MAX)|(MIN)|(MIN_ACTIVE_ROWVERSION)|(MONTH)|(NCHAR)|(NEWID)|(NEWSEQUENTIALID)|(NTILE)|(NULLIF)|(OBJECT_DEFINITION)|(OBJECT_ID)|(OBJECT_NAME)|(OBJECT_SCHEMA_NAME)|(OBJECTPROPERTY)|(OBJECTPROPERTYEX)|(OPENDATASOURCE)|(OPENQUERY)|(OPENROWSET)|(OPENXML)|(ORIGINAL_DB_NAME)|(ORIGINAL_LOGIN)|(PARSENAME)|(PathName)|(PATINDEX)|(PERMISSIONS)|(PI)|(POWER)|(PUBLISHINGSERVERNAME)|(PWDCOMPARE)|(PWDENCRYPT)|(QUOTENAME)|(RADIANS)|(RAISERROR)|(RAND)|(RANK)|(READTEXT)|(RECEIVE)|(RECONFIGURE)|(REPLACE)|(REPLICATE)|(REVERSE)|(RIGHT)|(ROUND)|(ROW_NUMBER)|(ROWCOUNT_BIG)|(RTRIM)|(SCHEMA_ID)|(SCHEMA_NAME)|(SCOPE_IDENTITY)|(SERVERPROPERTY)|(SESSION_USER)|(SESSIONPROPERTY)|(SETUSER)|(SIGN)|(SignByAsymKey)|(SignByCert)|(SIN)|(SOUNDEX)|(SPACE)|(SQL_VARIANT_PROPERTY)|(SQRT)|(SQUARE)|(STATS_DATE)|(STDEV)|(STDEVP)|(STR)|(STUFF)|(SUBSTRING)|(SUM)|(SUSER_ID)|(SUSER_NAME)|(SUSER_SID)|(SUSER_SNAME)|(SWITCHOFFSET)|(SYSDATETIME)|(SYSDATETIMEOFFSET)|(SYMKEYPROPERTY)|(SYSUTCDATETIME)|(SYSTEM_USER)|(TAN)|(TERTIARY_WEIGHTS)|(TEXTPTR)|(TEXTVALID)|(TODATETIMEOFFSET)|(TRIGGER_NESTLEVEL)|(TYPE_ID)|(TYPE_NAME)|(TYPEPROPERTY)|(UNICODE)|(UPDATE)|(UPPER)|(USER)|(USER_ID)|(USER_NAME)|(VARP)|(VerifySignedByCert)|(VerifySignedByAsymKey)|(xml)|(xml_schema_namespace)|(YEAR))(\s*\(|$)/gi; // collisions: IDENTITY
jush.tr.sql = { one: /-- |#|--(?=\n|$)/, com_code: /\/\*![0-9]*|\*\//, com: /\/\*/, sql_sqlset: /(\s*)(SET)(\s+|$)(?!NAMES\b|CHARACTER\b|PASSWORD\b|(?:GLOBAL\s+|SESSION\s+)?TRANSACTION\b|@[^@]|NEW\.|OLD\.)/i, sql_code: /()/ };
jush.tr.sql_code = { sql_apo: /'/, sql_quo: /"/, bac: /`/, one: /-- |#|--(?=\n|$)/, com_code: /\/\*![0-9]*|\*\//, com: /\/\*/, sql_var: /\B@/, num: jush.num, _1: /;|\b(THEN|ELSE|LOOP|REPEAT|DO)\b/i };
jush.tr.sql_sqlset = { one: /-- |#|--(?=\n|$)/, com: /\/\*/, sqlset_val: /=/, _1: /;|$/ };
jush.tr.sqlset_val = { sql_apo: /'/, sql_quo: /"/, bac: /`/, one: /-- |#|--(?=\n|$)/, com: /\/\*/, _1: /,/, _2: /;|$/, num: jush.num }; //! comma can be inside function call
jush.tr.sqlset = { _0: /$/ }; //! jump from SHOW VARIABLES LIKE ''
jush.tr.sqlstatus = { _0: /$/ }; //! jump from SHOW STATUS LIKE ''
jush.tr.com_code = { _1: /()/ };

jush.urls.sql_sqlset = 'https://dev.mysql.com/doc/mysql/en/$key';
jush.urls.sql = ['https://dev.mysql.com/doc/mysql/en/$key',
	'alter-event.html', 'alter-table.html', 'alter-view.html', 'analyze-table.html', 'create-event.html', 'create-function.html', 'create-procedure.html', 'create-index.html', 'create-table.html', 'create-trigger.html', 'create-view.html', 'drop-index.html', 'drop-table.html', 'begin-end.html', 'optimize-table.html', 'repair-table.html', 'set-transaction.html', 'show-columns.html', 'show-engines.html', 'show-index.html', 'show-processlist.html', 'show-status.html', 'show-tables.html', 'show-variables.html',
	'$1.html', '$1-statement.html', 'if-statement.html', 'repeat-statement.html', 'truncate-table.html', 'commit.html', 'savepoints.html', 'lock-tables.html', 'charset-connection.html', 'insert-on-duplicate.html', 'fulltext-search.html', 'example-auto-increment.html',
	'comparison-operators.html#operator_$1', 'comparison-operators.html#function_$1', 'any-in-some-subqueries.html', 'all-subqueries.html', 'exists-and-not-exists-subqueries.html', 'group-by-modifiers.html', 'string-functions.html#operator_$1', 'string-comparison-functions.html#operator_$1', 'regexp.html#operator_$1', 'regexp.html#operator_regexp', 'logical-operators.html#operator_$1', 'control-flow-functions.html#operator_$1', 'arithmetic-functions.html#operator_$1', 'cast-functions.html#operator_$1', 'date-and-time-functions.html#function_$1', 'date-and-time-functions.html#function_date-add',
	'', // keywords without link
	'numeric-type-syntax.html', 'date-and-time-type-syntax.html', 'string-type-syntax.html', 'mysql-spatial-datatypes.html',
	'mathematical-functions.html#function_$1', 'information-functions.html#function_$1',
	'$1-storage-engine.html', 'merge-storage-engine.html',
	'partitioning-range.html', 'partitioning-list.html', 'partitioning-columns.html', 'partitioning-hash.html', 'partitioning-linear-hash.html', 'partitioning-key.html',
	'comparison-operators.html#function_$1', 'control-flow-functions.html#function_$1', 'string-functions.html#function_$1', 'string-comparison-functions.html#function_$1', 'mathematical-functions.html#function_$1', 'date-and-time-functions.html#function_$1', 'cast-functions.html#function_$1', 'xml-functions.html#function_$1', 'bit-functions.html#function_$1', 'encryption-functions.html#function_$1', 'information-functions.html#function_$1', 'miscellaneous-functions.html#function_$1', 'group-by-functions.html#function_$1',
	'functions-to-convert-geometries-between-formats.html#function_asbinary',
	'functions-to-convert-geometries-between-formats.html#function_astext',
	'functions-for-testing-spatial-relations-between-geometric-objects.html#function_$1',
	'functions-that-create-new-geometries-from-existing-ones.html#function_$1',
	'geometry-property-functions.html#function_$1',
	'gis-wkt-functions.html#function_st-$1',
	'row-subqueries.html',
	'fulltext-search.html#function_match'
];
jush.urls.sqlset = ['https://dev.mysql.com/doc/mysql/en/$key',
	'innodb-parameters.html#sysvar_$1',
	'mysql-cluster-program-options-mysqld.html#option_mysqld_$1', 'mysql-cluster-replication-conflict-resolution.html#option_mysqld_$1', 'mysql-cluster-replication-schema.html', 'mysql-cluster-replication-starting.html', 'mysql-cluster-system-variables.html#sysvar_$1',
	'replication-options-binary-log.html#option_mysqld_$1', 'replication-options-binary-log.html#sysvar_$1', 'replication-options-master.html#sysvar_$1', 'replication-options-slave.html#option_mysqld_log-slave-updates', 'replication-options-slave.html#option_mysqld_$1', 'replication-options-slave.html#sysvar_$1', 'replication-options.html#option_mysqld_$1',
	'server-options.html#option_mysqld_big-tables', 'server-options.html#option_mysqld_$1',
	'server-system-variables.html#sysvar_$1', // previously server-session-variables
	'server-system-variables.html#sysvar_low_priority_updates', 'server-system-variables.html#sysvar_max_join_size', 'server-system-variables.html#sysvar_$1',
	'ssl-options.html#option_general_$1'
];
jush.urls.sqlstatus = ['https://dev.mysql.com/doc/mysql/en/$key',
	'server-status-variables.html#statvar_Com_xxx',
	'server-status-variables.html#statvar_$1'
];

jush.links.sql_sqlset = { 'set-statement.html': /.+/ };

jush.links2.sql = /(\b)(ALTER(?:\s+DEFINER\s*=\s*\S+)?\s+EVENT|(ALTER(?:\s+ONLINE|\s+OFFLINE)?(?:\s+IGNORE)?\s+TABLE)|(ALTER(?:\s+ALGORITHM\s*=\s*(?:UNDEFINED|MERGE|TEMPTABLE))?(?:\s+DEFINER\s*=\s*\S+)?(?:\s+SQL\s+SECURITY\s+(?:DEFINER|INVOKER))?\s+VIEW)|(ANALYZE(?:\s+NO_WRITE_TO_BINLOG|\s+LOCAL)?\s+TABLE)|(CREATE(?:\s+DEFINER\s*=\s*\S+)?\s+EVENT)|(CREATE(?:\s+DEFINER\s*=\s*\S+)?\s+FUNCTION)|(CREATE(?:\s+DEFINER\s*=\s*\S+)?\s+PROCEDURE)|(CREATE(?:\s+ONLINE|\s+OFFLINE)?(?:\s+UNIQUE|\s+FULLTEXT|\s+SPATIAL)?\s+INDEX)|(CREATE(?:\s+TEMPORARY)?\s+TABLE)|(CREATE(?:\s+DEFINER\s*=\s*\S+)?\s+TRIGGER)|(CREATE(?:\s+OR\s+REPLACE)?(?:\s+ALGORITHM\s*=\s*(?:UNDEFINED|MERGE|TEMPTABLE))?(?:\s+DEFINER\s*=\s*\S+)?(?:\s+SQL\s+SECURITY\s+(?:DEFINER|INVOKER))?\s+VIEW)|(DROP(?:\s+ONLINE|\s+OFFLINE)?\s+INDEX)|(DROP(?:\s+TEMPORARY)?\s+TABLE)|(END)|(OPTIMIZE(?:\s+NO_WRITE_TO_BINLOG|\s+LOCAL)?\s+TABLE)|(REPAIR(?:\s+NO_WRITE_TO_BINLOG|\s+LOCAL)?\s+TABLE)|(SET(?:\s+GLOBAL|\s+SESSION)?\s+TRANSACTION\s+ISOLATION\s+LEVEL)|(SHOW(?:\s+FULL)?\s+COLUMNS)|(SHOW(?:\s+STORAGE)?\s+ENGINES)|(SHOW\s+(?:INDEX|INDEXES|KEYS))|(SHOW(?:\s+FULL)?\s+PROCESSLIST)|(SHOW(?:\s+GLOBAL|\s+SESSION)?\s+STATUS)|(SHOW(?:\s+FULL)?\s+TABLES)|(SHOW(?:\s+GLOBAL|\s+SESSION)?\s+VARIABLES)|(ALTER\s+(?:DATABASE|SCHEMA)|ALTER\s+LOGFILE\s+GROUP|ALTER\s+SERVER|ALTER\s+TABLESPACE|BACKUP\s+TABLE|CACHE\s+INDEX|CALL|CHANGE\s+MASTER\s+TO|CHECK\s+TABLE|CHECKSUM\s+TABLE|CREATE\s+(?:DATABASE|SCHEMA)|CREATE\s+LOGFILE\s+GROUP|CREATE\s+SERVER|CREATE\s+TABLESPACE|CREATE\s+USER|DELETE|DESCRIBE|DO|DROP\s+(?:DATABASE|SCHEMA)|DROP\s+EVENT|DROP\s+FUNCTION|DROP\s+PROCEDURE|DROP\s+LOGFILE\s+GROUP|DROP\s+SERVER|DROP\s+TABLESPACE|DROP\s+TRIGGER|DROP\s+USER|DROP\s+VIEW|EXPLAIN|FLUSH|GRANT|HANDLER|HELP|INSERT|INSTALL\s+PLUGIN|JOIN|KILL|LOAD\s+DATA\s+FROM\s+MASTER|LOAD\s+DATA|LOAD\s+INDEX|LOAD\s+XML|PURGE\s+MASTER\s+LOGS|RENAME\s+(?:DATABASE|SCHEMA)|RENAME\s+TABLE|RENAME\s+USER|REPLACE|RESET\s+MASTER|RESET\s+SLAVE|RESIGNAL|RESTORE\s+TABLE|REVOKE|SELECT|SET\s+PASSWORD|SHOW\s+AUTHORS|SHOW\s+BINARY\s+LOGS|SHOW\s+BINLOG\s+EVENTS|SHOW\s+CHARACTER\s+SET|SHOW\s+COLLATION|SHOW\s+CONTRIBUTORS|SHOW\s+CREATE\s+(?:DATABASE|SCHEMA)|SHOW\s+CREATE\s+TABLE|SHOW\s+CREATE\s+VIEW|SHOW\s+(?:DATABASE|SCHEMA)S|SHOW\s+ENGINE|SHOW\s+ERRORS|SHOW\s+GRANTS|SHOW\s+MASTER\s+STATUS|SHOW\s+OPEN\s+TABLES|SHOW\s+PLUGINS|SHOW\s+PRIVILEGES|SHOW\s+SCHEDULER\s+STATUS|SHOW\s+SLAVE\s+HOSTS|SHOW\s+SLAVE\s+STATUS|SHOW\s+TABLE\s+STATUS|SHOW\s+TRIGGERS|SHOW\s+WARNINGS|SHOW|SIGNAL|START\s+SLAVE|STOP\s+SLAVE|UNINSTALL\s+PLUGIN|UNION|UPDATE|USE)|(LOOP|LEAVE|ITERATE|WHILE)|(IF|ELSEIF)|(REPEAT|UNTIL)|(TRUNCATE(?:\s+TABLE)?)|(START\s+TRANSACTION|BEGIN|COMMIT|ROLLBACK)|(SAVEPOINT|ROLLBACK\s+TO\s+SAVEPOINT)|((?:UN)?LOCK\s+TABLES?)|(SET\s+NAMES|SET\s+CHARACTER\s+SET)|(ON\s+DUPLICATE\s+KEY\s+UPDATE)|(IN\s+BOOLEAN\s+MODE|IN\s+NATURAL\s+LANGUAGE\s+MODE|WITH\s+QUERY\s+EXPANSION)|(AUTO_INCREMENT)|(IS|IS\s+NULL)|(BETWEEN|NOT\s+BETWEEN|IN|NOT\s+IN)|(ANY|SOME)|(ALL)|(EXISTS|NOT\s+EXISTS)|(WITH\s+ROLLUP)|(SOUNDS\s+LIKE)|(LIKE|NOT\s+LIKE)|(NOT\s+REGEXP|REGEXP)|(RLIKE)|(NOT|AND|OR|XOR)|(CASE)|(DIV)|(BINARY)|(CURRENT_DATE|CURRENT_TIME|CURRENT_TIMESTAMP|LOCALTIME|LOCALTIMESTAMP|UTC_DATE|UTC_TIME|UTC_TIMESTAMP)|(INTERVAL)|(ACCESSIBLE|ADD|ALTER|ANALYZE|AS|ASC|ASENSITIVE|BEFORE|BOTH|BY|CASCADE|CHANGE|CHARACTER|CHECK|CLOSE|COLLATE|COLUMN|CONDITION|CONSTRAINT|CONTINUE|CONVERT|CREATE|CROSS|CURSOR|DATABASE|DATABASES|DAY_HOUR|DAY_MICROSECOND|DAY_MINUTE|DAY_SECOND|DECLARE|DEFAULT|DELAYED|DESC|DETERMINISTIC|DISTINCT|DISTINCTROW|DROP|DUAL|EACH|ELSE|ENCLOSED|ESCAPED|EXIT|FALSE|FETCH|FLOAT4|FLOAT8|FOR|FORCE|FOREIGN|FROM|FULLTEXT|GROUP|HAVING|HIGH_PRIORITY|HOUR_MICROSECOND|HOUR_MINUTE|HOUR_SECOND|IGNORE|INDEX|INFILE|INNER|INOUT|INSENSITIVE|INT1|INT2|INT3|INT4|INT8|INTO|KEY|KEYS|LEADING|LEFT|LIMIT|LINEAR|LINES|LOAD|LOCK|LONG|LOW_PRIORITY|MASTER_SSL_VERIFY_SERVER_CERT|MATCH|MIDDLEINT|MINUTE_MICROSECOND|MINUTE_SECOND|MODIFIES|NATURAL|NO_WRITE_TO_BINLOG|NULL|OFFSET|ON|OPEN|OPTIMIZE|OPTION|OPTIONALLY|ORDER|OUT|OUTER|OUTFILE|PRECISION|PRIMARY|PROCEDURE|PURGE|RANGE|READ|READS|READ_WRITE|REFERENCES|RELEASE|RENAME|REQUIRE|RESTRICT|RETURN|RIGHT|SCHEMA|SCHEMAS|SECOND_MICROSECOND|SENSITIVE|SEPARATOR|SPATIAL|SPECIFIC|SQL|SQLEXCEPTION|SQLSTATE|SQLWARNING|SQL_BIG_RESULT|SQL_CALC_FOUND_ROWS|SQL_SMALL_RESULT|SSL|STARTING|STRAIGHT_JOIN|TABLE|TERMINATED|THEN|TO|TRAILING|TRIGGER|TRUE|UNDO|UNIQUE|UNLOCK|UNSIGNED|USAGE|USING|VALUES|VARCHARACTER|VARYING|WHEN|WHERE|WITH|WRITE|XOR|YEAR_MONTH|ZEROFILL))\b(?!\()|\b(bit|tinyint|bool|boolean|smallint|mediumint|int|integer|bigint|float|double\s+precision|double|real|decimal|dec|numeric|fixed|(date|datetime|timestamp|time|year)|(char|varchar|binary|varbinary|tinyblob|tinytext|blob|text|mediumblob|mediumtext|longblob|longtext|enum|set)|(geometry|point|linestring|polygon|multipoint|multilinestring|multipolygon|geometrycollection)|(mod)|(CURRENT_USER)|(InnoDB|MyISAM|MEMORY|CSV|ARCHIVE|BLACKHOLE|MERGE|FEDERATED)|(MRG_MyISAM)|(PARTITION\s+BY\s+RANGE)|(PARTITION\s+BY\s+LIST)|(PARTITION\s+BY\s+COLUMNS)|(PARTITION\s+BY\s+HASH)|(PARTITION\s+BY\s+LINEAR\s+HASH)|(PARTITION\s+BY(?:\s+LINEAR)?\s+KEY))\b|\b(coalesce|greatest|isnull|interval|least|(if|ifnull|nullif)|(ascii|bin|bit_length|char|char_length|character_length|concat|concat_ws|conv|elt|export_set|field|find_in_set|format|hex|insert|instr|lcase|left|length|load_file|locate|lower|lpad|ltrim|make_set|mid|oct|octet_length|ord|position|quote|repeat|replace|reverse|right|rpad|rtrim|soundex|sounds_like|space|substr|substring|substring_index|trim|ucase|unhex|upper)|(strcmp)|(abs|acos|asin|atan|atan2|ceil|ceiling|cos|cot|crc32|degrees|exp|floor|ln|log|log2|log10|pi|pow|power|radians|rand|round|sign|sin|sqrt|tan|truncate)|(adddate|addtime|convert_tz|curdate|curtime|date|datediff|date_add|date_format|date_sub|day|dayname|dayofmonth|dayofweek|dayofyear|extract|from_days|from_unixtime|get_format|hour|last_day|makedate|maketime|microsecond|minute|month|monthname|now|period_add|period_diff|quarter|second|sec_to_time|str_to_date|subdate|subtime|sysdate|time|timediff|timestamp|timestampadd|timestampdiff|time_format|time_to_sec|to_days|to_seconds|unix_timestamp|week|weekday|weekofyear|year|yearweek)|(cast|convert)|(extractvalue|updatexml)|(bit_count)|(aes_encrypt|aes_decrypt|compress|decode|encode|des_decrypt|des_encrypt|encrypt|md5|old_password|password|sha|sha1|uncompress|uncompressed_length)|(benchmark|charset|coercibility|collation|connection_id|database|found_rows|last_insert_id|row_count|schema|session_user|system_user|user|version)|(default|get_lock|inet_aton|inet_ntoa|is_free_lock|is_used_lock|master_pos_wait|name_const|release_lock|sleep|uuid|uuid_short|values)|(avg|bit_and|bit_or|bit_xor|count|count_distinct|group_concat|min|max|std|stddev|stddev_pop|stddev_samp|sum|var_pop|var_samp|variance)|(asbinary|aswkb)|(astext|aswkt)|(mbrcontains|mbrdisjoint|mbrequal|mbrintersects|mbroverlaps|mbrtouches|mbrwithin|contains|crosses|disjoint|equals|intersects|overlaps|touches|within)|(buffer|convexhull|difference|intersection|symdifference)|(dimension|envelope|geometrytype|srid|boundary|isempty|issimple|x|y|endpoint|glength|numpoints|pointn|startpoint|isring|isclosed|area|exteriorring|interiorringn|numinteriorrings|centroid|geometryn|numgeometries)|(geomcollfromtext|geomfromtext|linefromtext|mlinefromtext|mpointfromtext|mpolyfromtext|pointfromtext|polyfromtext|bdmpolyfromtext|bdpolyfromtext|geomcollfromwkb|geomfromwkb|linefromwkb|mlinefromwkb|mpointfromwkb|mpolyfromwkb|pointfromwkb|polyfromwkb|bdmpolyfromwkb|bdpolyfromwkb|geometrycollection|linestring|multilinestring|multipoint|multipolygon|point|polygon)|(row)|(match|against))(\s*\(|$)/gi; // collisions: char, set, union(), allow parenthesis - IN, ANY, ALL, SOME, NOT, AND, OR, XOR
jush.links2.sqlset = /(\b)(ignore_builtin_innodb|innodb_adaptive_hash_index|innodb_additional_mem_pool_size|innodb_autoextend_increment|innodb_autoinc_lock_mode|innodb_buffer_pool_awe_mem_mb|innodb_buffer_pool_size|innodb_commit_concurrency|innodb_concurrency_tickets|innodb_data_file_path|innodb_data_home_dir|innodb_doublewrite|innodb_fast_shutdown|innodb_file_io_threads|innodb_file_per_table|innodb_flush_log_at_trx_commit|innodb_flush_method|innodb_force_recovery|innodb_checksums|innodb_lock_wait_timeout|innodb_locks_unsafe_for_binlog|innodb_log_arch_dir|innodb_log_archive|innodb_log_buffer_size|innodb_log_file_size|innodb_log_files_in_group|innodb_log_group_home_dir|innodb_max_dirty_pages_pct|innodb_max_purge_lag|innodb_mirrored_log_groups|innodb_open_files|innodb_rollback_on_timeout|innodb_stats_on_metadata|innodb_support_xa|innodb_sync_spin_loops|innodb_table_locks|innodb_thread_concurrency|innodb_thread_sleep_delay|innodb_use_legacy_cardinality_algorithm|(ndb[-_]batch[-_]size)|(ndb[-_]log[-_]update[-_]as[-_]write|ndb_log_updated_only)|(ndb_log_orig)|(slave[-_]allow[-_]batching)|(have_ndbcluster|multi_range_count|ndb_autoincrement_prefetch_sz|ndb_cache_check_time|ndb_extra_logging|ndb_force_send|ndb_use_copying_alter_table|ndb_use_exact_count|ndb_wait_connected)|(log[-_]bin[-_]trust[-_]function[-_]creators|log[-_]bin)|(binlog_cache_size|max_binlog_cache_size|max_binlog_size|sync_binlog)|(auto_increment_increment|auto_increment_offset)|(ndb_log_empty_epochs)|(log[-_]slave[-_]updates|report[-_]host|report[-_]password|report[-_]port|report[-_]user|slave[-_]net[-_]timeout|slave[-_]skip[-_]errors)|(init_slave|rpl_recovery_rank|slave_compressed_protocol|slave_exec_mode|slave_transaction_retries|sql_slave_skip_counter)|(master[-_]bind|slave[-_]load[-_]tmpdir|server[-_]id)|(sql_big_tables)|(basedir|big[-_]tables|binlog[-_]format|collation[-_]server|datadir|debug|delay[-_]key[-_]write|engine[-_]condition[-_]pushdown|event[-_]scheduler|general[-_]log|character[-_]set[-_]filesystem|character[-_]set[-_]server|character[-_]sets[-_]dir|init[-_]file|language|large[-_]pages|log[-_]error|log[-_]output|log[-_]queries[-_]not[-_]using[-_]indexes|log[-_]slow[-_]queries|log[-_]warnings|log|low[-_]priority[-_]updates|memlock|min[-_]examined[-_]row[-_]limit|old[-_]passwords|open[-_]files[-_]limit|pid[-_]file|port|safe[-_]show[-_]database|secure[-_]auth|secure[-_]file[-_]priv|skip[-_]external[-_]locking|skip[-_]networking|skip[-_]show[-_]database|slow[-_]query[-_]log|socket|sql[-_]mode|tmpdir|version)|(autocommit|error_count|foreign_key_checks|identity|insert_id|last_insert_id|profiling|profiling_history_size|rand_seed1|rand_seed2|sql_auto_is_null|sql_big_selects|sql_buffer_result|sql_log_bin|sql_log_off|sql_log_update|sql_notes|sql_quote_show_create|sql_safe_updates|sql_warnings|timestamp|unique_checks|warning_count)|(sql_low_priority_updates)|(sql_max_join_size)|(automatic_sp_privileges|back_log|bulk_insert_buffer_size|collation_connection|collation_database|completion_type|concurrent_insert|connect_timeout|date_format|datetime_format|default_week_format|delayed_insert_limit|delayed_insert_timeout|delayed_queue_size|div_precision_increment|expire_logs_days|flush|flush_time|ft_boolean_syntax|ft_max_word_len|ft_min_word_len|ft_query_expansion_limit|ft_stopword_file|general_log_file|group_concat_max_len|have_archive|have_blackhole_engine|have_compress|have_crypt|have_csv|have_dynamic_loading|have_example_engine|have_federated_engine|have_geometry|have_innodb|have_isam|have_merge_engine|have_openssl|have_partitioning|have_query_cache|have_raid|have_row_based_replication|have_rtree_keys|have_ssl|have_symlink|hostname|character_set_client|character_set_connection|character_set_database|character_set_results|character_set_system|init_connect|interactive_timeout|join_buffer_size|keep_files_on_create|key_buffer_size|key_cache_age_threshold|key_cache_block_size|key_cache_division_limit|large_page_size|lc_time_names|license|local_infile|locked_in_memory|log_bin|long_query_time|lower_case_file_system|lower_case_table_names|max_allowed_packet|max_connect_errors|max_connections|max_delayed_threads|max_error_count|max_heap_table_size|max_insert_delayed_threads|max_join_size|max_length_for_sort_data|max_prepared_stmt_count|max_relay_log_size|max_seeks_for_key|max_sort_length|max_sp_recursion_depth|max_tmp_tables|max_user_connections|max_write_lock_count|myisam_data_pointer_size|myisam_max_sort_file_size|myisam_recover_options|myisam_repair_threads|myisam_sort_buffer_size|myisam_stats_method|myisam_use_mmap|named_pipe|net_buffer_length|net_read_timeout|net_retry_count|net_write_timeout|new|old|optimizer_prune_level|optimizer_search_depth|optimizer_switch|plugin_dir|preload_buffer_size|prepared_stmt_count|protocol_version|pseudo_thread_id|query_alloc_block_size|query_cache_limit|query_cache_min_res_unit|query_cache_size|query_cache_type|query_cache_wlock_invalidate|query_prealloc_size|range_alloc_block_size|read_buffer_size|read_only|read_rnd_buffer_size|relay_log_purge|relay_log_space_limit|shared_memory|shared_memory_base_name|slow_launch_time|slow_query_log_file|sort_buffer_size|sql_select_limit|storage_engine|sync_frm|system_time_zone|table_cache|table_definition_cache|table_lock_wait_timeout|table_open_cache|table_type|thread_cache_size|thread_concurrency|thread_handling|thread_stack|time_format|time_zone|timed_mutexes|tmp_table_size|transaction_alloc_block_size|transaction_prealloc_size|tx_isolation|updatable_views_with_limit|version_comment|version_compile_machine|version_compile_os|wait_timeout)|(ssl[-_]ca|ssl[-_]capath|ssl[-_]cert|ssl[-_]cipher|ssl[-_]key))((?!-)\b)/gi;
jush.links2.sqlstatus = /()(Com_.+|(.+))()/gi;
jush.tr.sqlite = { sqlite_apo: /'/, sqlite_quo: /"/, bra: /\[/, bac: /`/, one: /--/, com: /\/\*/, sql_var: /[:@$]/, sqlite_sqliteset: /(\b)(PRAGMA)(\s+)/i, num: jush.num };
jush.tr.sqlite_sqliteset = { sqlite_apo: /'/, sqlite_quo: /"/, bra: /\[/, bac: /`/, one: /--/, com: /\/\*/, num: jush.num, _1: /;|$/ };
jush.tr.sqliteset = { _0: /$/ };
jush.tr.sqlitestatus = { _0: /$/ };

jush.urls.sqlite_sqliteset = 'https://www.sqlite.org/$key';
jush.urls.sqlite = ['https://www.sqlite.org/$key',
	'lang_$1.html', 'lang_createvtab.html', 'lang_transaction.html',
	'lang_createindex.html', 'lang_createtable.html', 'lang_createtrigger.html', 'lang_createview.html',
	'',
	'lang_expr.html#$1', 'lang_corefunc.html#$1', 'lang_datefunc.html#$1', 'lang_aggfunc.html#$1'
];
jush.urls.sqliteset = ['https://www.sqlite.org/pragma.html#$key',
	'pragma_$1'
];
jush.urls.sqlitestatus = ['https://www.sqlite.org/compile.html#$key',
	'$1'
];

jush.links.sqlite_sqliteset = { 'pragma.html': /.+/ };

jush.links2.sqlite = /(\b)(ALTER\s+TABLE|ANALYZE|ATTACH|COPY|DELETE|DETACH|DROP\s+INDEX|DROP\s+TABLE|DROP\s+TRIGGER|DROP\s+VIEW|EXPLAIN|INSERT|CONFLICT|REINDEX|REPLACE|SELECT|UPDATE|TRANSACTION|VACUUM|(CREATE\s+VIRTUAL\s+TABLE)|(BEGIN|COMMIT|ROLLBACK)|(CREATE(?:\s+UNIQUE)?\s+INDEX)|(CREATE(?:\s+TEMP|\s+TEMPORARY)?\s+TABLE)|(CREATE(?:\s+TEMP|\s+TEMPORARY)?\s+TRIGGER)|(CREATE(?:\s+TEMP|\s+TEMPORARY)?\s+VIEW)|(ABORT|ACTION|ADD|AFTER|ALL|AS|ASC|AUTOINCREMENT|BEFORE|BY|CASCADE|CHECK|COLUMN|CONSTRAINT|CROSS|CURRENT_DATE|CURRENT_TIME|CURRENT_TIMESTAMP|DATABASE|DEFAULT|DEFERRABLE|DEFERRED|DESC|DISTINCT|EACH|END|EXCEPT|EXCLUSIVE|FAIL|FOR|FOREIGN|FROM|FULL|GROUP|HAVING|IF|IGNORE|IMMEDIATE|INDEXED|INITIALLY|INNER|INSTEAD|INTERSECT|INTO|IS|JOIN|KEY|LEFT|LIMIT|NATURAL|NO|NOT|NOTNULL|NULL|OF|OFFSET|ON|ORDER|OUTER|PLAN|PRAGMA|PRIMARY|QUERY|RAISE|REFERENCES|RELEASE|RENAME|RESTRICT|RIGHT|ROW|SAVEPOINT|SET|TEMPORARY|TO|UNION|UNIQUE|USING|VALUES|WHERE)|(like|glob|regexp|match|escape|isnull|isnotnull|between|exists|case|when|then|else|cast|collate|in|and|or|not))\b|\b(abs|coalesce|glob|ifnull|hex|last_insert_rowid|length|like|load_extension|lower|nullif|quote|random|randomblob|round|soundex|sqlite_version|substr|typeof|upper|(date|time|datetime|julianday|strftime)|(avg|count|max|min|sum|total))(\s*\(|$)/gi; // collisions: min, max, end, like, glob
jush.links2.sqliteset = /(\b)(auto_vacuum|cache_size|case_sensitive_like|count_changes|default_cache_size|empty_result_callbacks|encoding|foreign_keys|full_column_names|fullfsync|incremental_vacuum|journal_mode|journal_size_limit|legacy_file_format|locking_mode|page_size|max_page_count|read_uncommitted|recursive_triggers|reverse_unordered_selects|secure_delete|short_column_names|synchronous|temp_store|temp_store_directory|collation_list|database_list|foreign_key_list|freelist_count|index_info|index_list|page_count|table_info|schema_version|compile_options|integrity_check|quick_check|parser_trace|vdbe_trace|vdbe_listing)(\b)/gi;
jush.links2.sqlitestatus = /()(.+)()/g;
